from django.contrib.auth import get_user_model
from django.contrib.auth.models import AnonymousUser
from django.core.signals import request_finished
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers
from rest_framework.serializers import Serializer
from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.intranet.models.consts import AFFILIATION
from wiki.intranet.models.staff import Staff
from wiki.org import org_staff
from wiki.pages.access import is_admin
from wiki.users import DEFAULT_AVATAR_ID
from wiki.users.dao import get_user_by_identity, UserNotFound
from wiki.users.logic import settings as settings_logic
from wiki.utils.features.get_features import get_wiki_features
from wiki.utils.injector import thread_locals

# кэш объктов Staff, ключ - user.id (staff.user_id)
thread_locals._staff_cache = {}

User = get_user_model()


class UserProxy(object):
    """
    Инкапсуляция users.User для API
    Создает из объекта users.User объект с полями, которые нужны в API.
    """

    def __init__(self, user):
        """

        @type user: джанго User
        """
        # работаем лишь с юзерами джанги
        if not isinstance(user, (User, AnonymousUser)):
            raise ValueError(f'user object ({user}, {user.__class__}) must be an instance of some User class')

        is_guest = isinstance(user, AnonymousUser)

        # ищем профиль - объект Staff
        # user.id == staff.user_id
        staff = thread_locals._staff_cache.get(user.id, None)
        if not staff:
            try:
                staff = user.staff
                thread_locals._staff_cache[user.id] = staff
            except (Staff.DoesNotExist, AttributeError):
                # У некоторых RPC-пользователей и AnonymousUser нет профайла
                pass

        self.login = user.username

        self.uid = staff and int(staff.uid)

        self.cloud_uid = user.cloud_uid if not is_guest else None

        self.is_dismissed = staff and staff.is_dismissed
        self.is_admin = is_admin(user)
        self.is_external_employee = staff and staff.affiliation == AFFILIATION.EXTERNAL

        self.settings = settings_logic.get_user_settings(user)

        self.display = staff and staff.get_full_name()
        self.first_name = staff and staff.i_first_name
        self.last_name = staff and staff.i_last_name
        self.email = staff and staff.work_email

        self.avatar_id = DEFAULT_AVATAR_ID
        self.css_theme = None

        if get_wiki_features().avatars_enabled:
            self.load_avatar(user)

    def load_avatar(self, user):
        if hasattr(user, 'avatar_id'):
            self.avatar_id = user.avatar_id

    @classmethod
    def populate_cache(cls, users):
        """
        Создать кэш из объектов User и Staff, относящихся к переданным
        usernames

        @param usernames: список логинов пользователей, для которых создается кэш
        """
        staff = org_staff().filter(user__in=[user.id for user in users])
        staff_list = list(staff)

        for staff_obj in staff_list:
            thread_locals._staff_cache[staff_obj.user_id] = staff_obj

    @classmethod
    def clear_cache(self, *args, **kwargs):
        """
        Очистить кэши
        *args, **kwargs нужны, ибо этот метод может быть повешен на сигнал
        """
        thread_locals._staff_cache = {}

    # чистим кэш юзеров, вики-юзеров и стаффа в конце запроса, чтобы


# он не использовался при обработке следующих - ибо в базе объекты могли измениться
request_finished.connect(UserProxy.clear_cache)


class UserListItemSerializer(serializers.Serializer):
    uid = serializers.IntegerField(default=None)
    cloud_uid = serializers.CharField(default=None)
    first_name = serializers.CharField(read_only=True)
    last_name = serializers.CharField(read_only=True)
    login = serializers.CharField(read_only=True)
    avatar_id = serializers.CharField(read_only=True)


class UserSerializer(UserListItemSerializer):
    """
    Принимает объект User и сам достает из него и из профайла нужные поля.
    """

    display = serializers.CharField(read_only=True)
    email = serializers.EmailField(read_only=True)
    is_admin = serializers.BooleanField(read_only=True)
    is_external_employee = serializers.BooleanField(read_only=True)
    is_dismissed = serializers.BooleanField(read_only=True)

    default_error_messages = {
        'user_not_found': _('User with uid={uid} was not found'),
        'user_should_not_be_dismissed': _('User {login} is dismissed'),
        'user_should_be_dismissed': _('User {login} is not dismissed'),
    }

    def __init__(self, *args, **kwargs):
        self.should_be_dismissed = kwargs.pop('should_be_dismissed', None)
        super(UserSerializer, self).__init__(*args, **kwargs)

    def to_representation(self, instance):
        """
        Serialize objects -> primitives.
        """
        user_proxy = UserProxy(instance)

        return super(UserSerializer, self).to_representation(user_proxy)

    def run_validators(self, value):
        # Хак для того, чтобы этот код мог работать как сериализатор и как поле (что поперек архитектуры DRF
        # но уж так этот код был написан)

        # Дело в том, что to_internal_value у Сериализатора ожидает словарь на выходе,
        # и в run_validators этот словарь дополняется значениями по-умолчанию
        # Поле же может отдавать из to_internal_value что угодно (как тут)
        # Де-факто, мы вызовем тут Field.run_validators(value)
        super(Serializer, self).run_validators(value)

    def to_internal_value(self, data):
        uid = data.get('uid')
        cloud_uid = data.get('cloud_uid')

        try:
            user = get_user_by_identity(UserIdentity(uid=uid, cloud_uid=cloud_uid), prefetch_staff=True)
        except UserNotFound:
            self.fail('user_not_found', uid=cloud_uid or uid)

        if self.should_be_dismissed is not None:
            is_dismissed = user.staff.is_dismissed
            if is_dismissed and not self.should_be_dismissed:
                self.fail('user_should_not_be_dismissed', login=user.staff.login)
            if not is_dismissed and self.should_be_dismissed:
                self.fail('user_should_be_dismissed', login=user.staff.login)

        return user


class RequestUserSerializer(UserSerializer):
    """
    Пользователь, сделавший запрос. Отличается наличием поля settings
    """

    settings = serializers.ReadOnlyField()
