from functools import wraps
from urllib.parse import urlparse

from django.conf import settings
from django.http import QueryDict
from wiki.legacy.choices import Choices
from rest_framework.permissions import BasePermission

from wiki import access as wiki_access

# коды типов доступок к странице, используемые в API
from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.org import get_org
from wiki.pages.access import access_status as wiki_access_status
from wiki.pages.models import Page
from wiki.utils.supertag import translit

API_ACCESS_TYPES = Choices(
    # свободный сотрудникам Яндекса
    common=('common', wiki_access.TYPES.COMMON),
    # только авторам
    owner=('owner', wiki_access.TYPES.OWNER),
    # ограниченный по юзерам/группам
    restricted=('restricted', wiki_access.TYPES.RESTRICTED),
    # унаследованный от родителя
    inherited=('inherited', wiki_access.TYPES.INHERITED),
)

# коды доступности страницы для конкретного пользователя
API_USER_ACCESS = Choices(
    **{
        'nonexistent': wiki_access_status.ACCESS_NONEXISTENT,  # нет страницы
        'denied': wiki_access_status.ACCESS_DENIED,  # доступ пользователю запрещен
        'allowed-common': wiki_access_status.ACCESS_COMMON,
        # доступ пользователю разрешен (т.к. разрешен для яндексоидов)
        'allowed-restricted': wiki_access_status.ACCESS_RESTRICTED,  # доступ пользователю разрешен
        # (т.к. разрешен для отдельных групп/пользователей)
        'allowed-unlimited': wiki_access_status.ACCESS_UNLIMITED,  # доступ разрешен (т.к. разрешен всем)
    }
)


def is_tvm_authentication(request):
    return (
        hasattr(request, 'yauser')
        and request.yauser.authenticated_by
        and request.yauser.authenticated_by.mechanism_name == 'tvm'
    )


def is_oauth2_authentication(request):
    return (
        hasattr(request, 'yauser')
        and request.yauser.authenticated_by
        and request.yauser.authenticated_by.mechanism_name == 'oauth'
    )


def is_authenticated(request):
    return (
        hasattr(request, 'yauser')
        and request.yauser.authenticated_by
        and request.yauser.authenticated_by.mechanism_name in {'tvm', 'oauth'}
    )


def is_docviewer_previewable(file_obj):
    """
    Хелпер для определения возможности предпросмотра файла в docviewer
    """
    pos = file_obj.url.rfind('.')
    if pos > -1:
        extension = file_obj.url[pos + 1 :]
        return extension.lower() in settings.DOCVIEWER_PREVIEWABLE_FILE_TYPES
    return False


def paginated(method):
    """
    Декоратор для методов get/post/put/delete инстансов наследников APIView.
    Вынимает и валидирует start и limit параметры запросов с пагинацией.
    """

    @wraps(method)
    def wrapper(inst, *args, **kwargs):
        """
        @type ints: APIView
        """
        start = inst.request.GET.get('start', None)
        limit = inst.request.GET.get('limit', None)

        if start:
            start = int(start)

        if limit:
            limit = int(limit)

        kwargs.setdefault('start', start)
        kwargs.setdefault('limit', limit)

        return method(inst, *args, **kwargs)

    return wrapper


def parse_url(url):
    """Вернуть представление URL, удобное для маршрутизации урлов в вики.

    string -> {"url": string_without_first_slash, "handler_name": lowercased_string, "content_type": string|None}
    """
    tokens = [u for u in url.split('/') if u]
    handler_name = 'show'
    content_type = 'html'
    if tokens and tokens[-1].startswith('.'):
        handler_tokens = [h for h in tokens[-1].split('.') if h]
        if len(handler_tokens) > 1:
            handler_name = '.'.join(handler_tokens[:-1])
            content_type = handler_tokens[-1]
        else:
            handler_name = '.'.join(handler_tokens)
        url = '/'.join(tokens[:-1])
    else:
        url = '/'.join(tokens)
    url = url or settings.MAIN_PAGE
    return {'handler_name': handler_name.lower(), 'content_type': content_type, 'url': url}


def find_wiki_page_by_url(url):
    """
    Найти в базе данных вики страницу по переданному url.
    Если url соответствует вики странице, но по supertag страницы найти ее в базе невозможно,
    то вернуть значение supertag.

    @rtype: tuple
    @return: (supertag страницы или None, объект page или None)
    """
    supertag = None
    page = None
    if url:
        purl = urlparse(url)
        if purl.netloc in settings.FRONTEND_HOSTS:
            dispatch_vars = parse_url(purl.path)
            supertag = translit(dispatch_vars['url'])
            try:
                page = Page.objects.get(supertag=supertag, org=get_org())
            except Page.DoesNotExist:
                pass

    return supertag, page


def _get_bool(query_data, param_name, default=None):
    """
    Получить из dict-like объекта параметров django запроса значение по ключу {param_name}
    и привести к bool с учетом того, что в значении могут быть 'true' | 'false'
    из js.
    @type query_data: QueryDict
    @param query_data: параметры django запроса
    @type param_name: basestring
    @param param_name: имя приводимого к bool параметра
    @type default: bool
    @param default: дефолтное значение
    @rtype: bool
    @raise: InvalidDataSentError
    """
    assert isinstance(query_data, QueryDict)

    value = query_data.get(param_name, default)

    if isinstance(value, bool):
        return value

    if isinstance(value, str):
        if value.isdigit():
            return bool(int(value))

        elif value.lower() in ('true', 'false'):
            return value.lower() == 'true'

        elif default is not None:
            return default

    raise InvalidDataSentError


class UIDListPermission(BasePermission):
    """
    Базовый класс доступа для тех view, доступ к которым определяется на основе UID.
    Класс-наследник переопределяет uid_list - список UID пользователей, имеющих доступ к view.
    Если uid_list=None, то доступ имеют все.
    """

    uid_list = None
    cloud_uid_list = None

    def has_permission(self, request, view):
        if is_tvm_authentication(request):
            # это костыль на время перехода на TVM2 - при аутентификации по сервисному тикету проверка
            # будет осуществляться в TVMIDPermission классе.
            return True

        if self.uid_list and request.user.staff.uid in self.uid_list:
            return True

        if self.cloud_uid_list and request.user.cloud_uid in self.cloud_uid_list:
            return True

        if not self.cloud_uid_list and not self.uid_list:
            return True

        raise UserHasNoAccess('User uid not in allowed uid list')


class DirectoryRobotPermission(BasePermission):
    """
    Проверяет то, что данный пользователь Директории - роботный, и его service_slug.
    """

    service_slug = None

    def has_permission(self, request, view):
        if is_tvm_authentication(request):
            # это костыль на время перехода на TVM2 - при аутентификации по сервисному тикету проверка
            # будет осуществляться в TVMIDPermission классе.
            return True

        return request.user.is_dir_robot and request.user.service_slug == self.service_slug


class TVMIDPermission(BasePermission):
    """
    Проверяет, что в запросе был передан валидный TVM2 сервисный тикет
    и ID TVM2 приложения находится в списке разрешенных.
    """

    tvm_client_ids = None
    strict = False

    def has_permission(self, request, view):
        if not is_tvm_authentication(request):
            if not self.strict or not settings.STRICT_TVM2:
                # TODO удалить после перехода Поиска и СИБ на TVM2 в соответствующих ручках
                # Этот if блок нужен как костыль для поддержки устаревшего метода аутентификации по oauth токену,
                # при этом проверка токен происходит в другом BasePermission классе -
                # в UIDListPermission или в DirectoryRobotPermission, в зависимости от типа инстанса сервиса
                return True
            else:
                raise UserHasNoAccess('Only TVM authentication method is supported for this view.')

        if self.tvm_client_ids:
            if not is_tvm_authentication(request):
                raise UserHasNoAccess('Please use TVM2 auth for this view')
            if not str(request.yauser.service_ticket.src) in self.tvm_client_ids:
                raise UserHasNoAccess('Client id not in allowed id list')

        return True


class StrictTVMIDPermission(TVMIDPermission):
    strict = True


def get_nginx_host(request):
    """
    Вики может быть доступна по нескольким доменам, поэтому
    домен следует брать из оригинального заголовка Host,
    который прокидывается Nginx-ом под другим именем.
    В качестве fallback-а берется settings.NGINX_HOST.
    """
    return request.META.get(settings.NGINX_HOST_HEADER) or settings.NGINX_HOST
