"""
Генераторы для совместимости старого фронтэнда с новым.
"""
import logging
from functools import wraps

from django.conf import settings
from django.utils.translation import get_language

from wiki.api_core.waffle_switches import GRID_NO_WOF
from wiki.grids.utils.base import CHECKBOX_TYPE, HASH_KEY, SELECT_TYPE, STAFF_TYPE, TICKET_TYPE, ticket_field_names
from wiki.org import org_staff

logger = logging.getLogger(__name__)


def init_next_generator(func):
    @wraps(func)
    def do(next_gen=None, *args, **kwargs):
        if next_gen is not None:
            next_gen.send(None)
        return func(next_gen, *args, **kwargs)

    return do


@init_next_generator
def none_in_cells_gen(next_gen, grid):
    """
    Убрать null из json-представления грида.

    """
    # TODO: нужна миграция после отказа от старого фронтэнда, чтобы удалить этот метод.

    select_field_names = set(grid.column_names_by_type(SELECT_TYPE))
    staff_field_names = set(
        grid.column_names_by_type(STAFF_TYPE, ticket_field_names.assignee, ticket_field_names.reporter)
    )

    checkbox_field_names = set(grid.column_names_by_type(CHECKBOX_TYPE))

    while True:
        row = yield
        for cell_key in row:
            # исключить случаи, когда значение ячейки - это None или список с пустой строкой [""]
            if row[cell_key] is None or (
                cell_key in (select_field_names | staff_field_names)
                and len(row[cell_key]['raw']) == 1
                and not row[cell_key]['raw'][0]
            ):
                # переопределяем значения всех ячеек в зависимости от типа
                if cell_key in select_field_names:
                    # select содержащит список значений
                    row[cell_key] = {
                        'raw': [],
                    }
                elif cell_key in staff_field_names:
                    # для правильной сортировки при наличии пустых ячеек добавляем sort
                    row[cell_key] = {
                        'raw': [],
                        'sort': '',
                    }
                elif cell_key in checkbox_field_names:
                    # для checkbox
                    row[cell_key] = {
                        'raw': False,
                    }
                else:
                    # на самом деле это только для number
                    row[cell_key] = {
                        'raw': None,
                    }

        if next_gen:
            next_gen.send(row)


VIEW_CELL_NAME = 'view'
TRANSFORMED_CELL_NAME = 'transformed'


@init_next_generator
def add_row_id_to_cell_gen(next_gen, grid):
    """
    Генератор. В ячейки добавляется row_id - идентификатор строки, к которой относится ячейка
    @param next_gen: следующий генератор в цепочке, если есть
    @param grid: grid
    """
    while True:
        row = yield
        row_id = row[HASH_KEY]

        for cell_key in row.keys():
            if cell_key.startswith('__'):
                continue

            cell = row[cell_key]
            if not isinstance(cell, dict):
                raise ValueError('cell %s in grid %s row %s is not a dict', cell_key, str(grid), row_id)

            cell['row_id'] = row_id

        if next_gen:
            next_gen.send(row)


@init_next_generator
def text_transformed_to_view_gen(next_gen, grid):
    """
    Генератор. В ячейках с полем VIEW_CELL_NAME проставляет еще и TRANSFORMED_CELL_NAME
    """
    string_field_names = list(grid.column_names_by_type('string'))
    while True:
        row = yield
        for name in string_field_names:
            cell = row.get(name, None)
            if cell is None:
                continue
            if VIEW_CELL_NAME in cell:
                if not GRID_NO_WOF:
                    if TRANSFORMED_CELL_NAME in cell:
                        logger.warning(
                            'Overriding transformed field of grid "%s", row "%s", cell "%s"',
                            grid.supertag,
                            row[HASH_KEY],
                            name,
                        )
                    cell[TRANSFORMED_CELL_NAME] = cell[VIEW_CELL_NAME]
                del cell[VIEW_CELL_NAME]

        if next_gen:
            next_gen.send(row)


WF_VERSION = 'wf_version'


@init_next_generator
def no_wf_version(next_gen):
    """
    В гридах не бывает поля wf_version.
    """
    while True:
        row = yield

        for cell_key in row:
            if cell_key.startswith('__'):
                continue
            if WF_VERSION in row[cell_key]:
                del row[cell_key][WF_VERSION]

        if next_gen:
            next_gen.send(row)


def get_from_staff_names(staff_names, login, lang=''):
    """
    @return: tuple из логина, имени, фамилии, флага is_dismissed
    @rtype: tuple
    """
    if login not in staff_names:

        staff = org_staff().filter(login=login).order_by('is_dismissed').first()
        if staff is None:
            staff_names[login] = login, '', '', False
        else:
            safe_is_dismissed = getattr(staff, 'is_dismissed', False)
            lang_postfix = '_en' if lang == 'en' else ''
            first_name = getattr(staff, 'first_name%s' % lang_postfix)
            last_name = getattr(staff, 'last_name%s' % lang_postfix)
            cloud_uid = staff.user.cloud_uid

            staff_names[login] = login, first_name, last_name, safe_is_dismissed, cloud_uid

    return staff_names[login]


ZIPPED_SEQUENCE = (
    'login',
    'first_name',
    'last_name',
    'is_dismissed',
    'cloud_uid',
)


@init_next_generator
def staff_field(next_gen, grid):
    """
    Поле типа staff содержит transform в виде списка, не дикта.

    """
    staff_field_names = [
        field['name']
        for field in grid.access_structure['fields']
        if field['type'] in (STAFF_TYPE, ticket_field_names.assignee, ticket_field_names.reporter)
    ]

    staff_names = {}
    lang = get_language()

    # @fixme Refactor me, Luke

    while True:
        row = yield
        for name in staff_field_names:
            cell = row.get(name, None)
            if cell is None:
                continue

            transformed = []
            if cell['raw']:
                for login in cell['raw']:
                    if login:
                        transformed.append(
                            dict(list(zip(ZIPPED_SEQUENCE, get_from_staff_names(staff_names, login, lang))))
                        )
            cell['transformed'] = transformed

        if next_gen:
            next_gen.send(row)


@init_next_generator
def tracker_ticket_field(next_gen, grid):
    ticket_field_names = list(grid.column_names_by_type(TICKET_TYPE))

    while True:
        row = yield
        for name in ticket_field_names:
            cell = row.get(name, None)
            if cell is None:
                continue

            if 'view' in cell:
                del cell['view']

            if not cell.get('raw'):
                continue

            cell['url'] = '{0}/{1}'.format(settings.STARTREK_CONFIG['INTERFACE_URL'], cell['raw'])

        if next_gen:
            next_gen.send(row)


@init_next_generator
def cut_b64wom1(next_gen, grid):
    string_field_names = set(grid.column_names_by_type('string'))

    while True:
        row = yield

        for cell_key in string_field_names:
            cell = row.get(cell_key)

            if cell and 'b64wom1' in cell:
                del cell['b64wom1']

        if next_gen:
            next_gen.send(row)
