import logging

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.sync.connect.models import Organization
from wiki.sync.connect.tasks.consts import (
    CACHE_FORCED_SYNC_TIMEOUT,
    ForcedSyncStatus,
    ForcedSyncErrorCode,
    ForcedSyncResult,
)
from wiki.sync.connect.tasks.forced_sync import (
    get_cache_key,
    CACHE_FORCED_SYNC_BACKEND,
    ForcedSyncTask,
    ForcedSyncData,
)
from wiki.sync.connect.tasks.helpers import get_user_orgs_from_connect
from wiki.users.dao import get_user_by_identity, UserNotFound

logger = logging.getLogger(__name__)


def recover_state(dir_org_id, cloud_org_id, user_cloud_uid, user_uid) -> ForcedSyncResult | None:
    cache_key = get_cache_key(dir_org_id, cloud_org_id, user_cloud_uid, user_uid)
    task_result_dict = CACHE_FORCED_SYNC_BACKEND.get(cache_key)

    if task_result_dict:
        try:
            ret_obj = ForcedSyncResult.parse_obj(task_result_dict)
            return ret_obj
        except Exception:
            logger.exception(f'Sync result is {task_result_dict}; can not deserialize it')
            return ForcedSyncResult(result=ForcedSyncStatus.FAILED, error_code=ForcedSyncErrorCode.SERDES_ERROR)

    return None


def save_state(r, dir_org_id: str, cloud_org_id: str, user_cloud_uid: str, user_uid: str):
    cache_key = get_cache_key(dir_org_id, cloud_org_id, user_cloud_uid, user_uid)
    CACHE_FORCED_SYNC_BACKEND.set(cache_key, r.dict(), CACHE_FORCED_SYNC_TIMEOUT)


def drop_state(user):
    orgs = user.orgs.all()
    for org in orgs:
        cache_key = get_cache_key(org.dir_id, org.cloud_id or '', user.cloud_uid, user.dir_id)
        CACHE_FORCED_SYNC_BACKEND.delete(cache_key)


def check_if_synced(data: ForcedSyncData) -> bool:
    identity = UserIdentity(uid=data.user_uid, cloud_uid=data.user_cloud_uid)
    try:
        user = get_user_by_identity(identity, limit_users_by_current_org=False)
    except UserNotFound:
        return False

    user_orgs = set(user.orgs.all())
    if not user_orgs:
        return False

    # ищем организацию, если передали id
    if data.dir_org_id or data.cloud_org_id:
        found_org = None
        for org in user_orgs:
            if org.dir_id == data.dir_org_id or org.cloud_id == data.cloud_org_id:
                found_org = org

        if found_org and found_org.status == Organization.ORG_STATUSES.enabled:
            data.dir_org_id = found_org.dir_id
            return True

        return False

    # проверяем что есть организации в коннекте и все они есть в бд
    remote_orgs = get_user_orgs_from_connect(user_uid=data.user_uid, user_cloud_uid=data.user_cloud_uid)
    if not remote_orgs:
        return False

    if {remote_org['id'] for remote_org in remote_orgs} == {org.dir_id for org in user_orgs}:
        data.dir_org_id = remote_orgs[0]['id']
        return True

    return False


def forced_sync(
    dir_org_id: str,
    cloud_org_id: str,
    user_cloud_uid: str,
    user_uid: str,
    user_iam_token: str,
    skip_check: bool = False,
) -> ForcedSyncResult:
    """
    Проверить наличие в нашей базе организации с переданным dir_org_id или cloud_org_id

    И наличие пользователя с переданным cloud_uid или uid в этой организации
    """

    sync_data = ForcedSyncData(
        dir_org_id=dir_org_id,
        cloud_org_id=cloud_org_id,
        user_cloud_uid=user_cloud_uid,
        user_uid=user_uid,
        user_iam_token=user_iam_token,
    )

    if not skip_check and check_if_synced(sync_data):
        logger.info(f'Will not schedule forced sync for ${sync_data.secure_dict()}; as it is seems to be alright')
        return ForcedSyncResult(result=ForcedSyncStatus.OK, org_id=sync_data.dir_org_id)

    state = recover_state(dir_org_id, cloud_org_id, user_cloud_uid, user_uid)

    if state and not skip_check:
        if state.result != ForcedSyncStatus.OK:
            return state
        else:
            # явная проверка в начале функции сказала что с пользователем что-то не то.
            # скорее всего кеш тут устарел
            logger.info(
                'Having obsolete sync ok status in cache '
                + f'but check is failed ({dir_org_id}/{cloud_org_id}/{user_cloud_uid}/{user_uid})'
            )

    sync_data = ForcedSyncData(
        dir_org_id=dir_org_id,
        cloud_org_id=cloud_org_id,
        user_cloud_uid=user_cloud_uid,
        user_uid=user_uid,
        user_iam_token=user_iam_token,
    )

    r = ForcedSyncResult(result=ForcedSyncStatus.IN_PROGRESS)
    save_state(r, dir_org_id, cloud_org_id, user_cloud_uid, user_uid)
    ForcedSyncTask().delay(**sync_data.dict())

    logger.info(f'Scheduled forced sync for ${sync_data.secure_dict()}')
    return r


def forced_sync_now(
    dir_org_id: str,
    cloud_org_id: str,
    user_cloud_uid: str,
    user_uid: str,
    user_iam_token: str,
) -> ForcedSyncResult:
    cache_key = get_cache_key(dir_org_id, cloud_org_id, user_cloud_uid, user_uid)

    sync_data = ForcedSyncData(
        dir_org_id=dir_org_id,
        cloud_org_id=cloud_org_id,
        user_cloud_uid=user_cloud_uid,
        user_uid=user_uid,
        user_iam_token=user_iam_token,
    )

    r = ForcedSyncResult(result=ForcedSyncStatus.IN_PROGRESS)
    CACHE_FORCED_SYNC_BACKEND.set(cache_key, r.dict(), CACHE_FORCED_SYNC_TIMEOUT)

    ForcedSyncTask().run(**sync_data.dict())
    return r
