
import logging

from cached_property import cached_property
from django.utils.translation import ugettext as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.hacks.grid_hacks import replace_width_in_grid_structure
from wiki.api_frontend.logic.data_generators import (
    add_row_id_to_cell_gen,
    cut_b64wom1,
    no_wf_version,
    none_in_cells_gen,
    staff_field,
    text_transformed_to_view_gen,
    tracker_ticket_field,
)
from wiki.grids.filter import FilterError, assert_columns_exist, filter_grid, get_column_names, get_filter_parts
from wiki.grids.filter import parse as parse_filter
from wiki.grids.logic.operations import sort_grid_rows
from wiki.grids.utils.base import HASH_KEY

log = logging.getLogger(__name__)


class GridProxy(object):
    """
    Преобразует данные из формата в базе в формат для nodejs-фронтэнда.
    (они там лежат в удобном для старой верстки виде).
    """

    _grid = None
    __sorting = None
    # направления, доступные для сортировки
    AVAILABLE_SORTS = ('desc', 'asc')
    SORT_KEYWORD = 'sort_'
    FILTER_PARAM_NAME = 'filter'
    COLUMNS_PARAM_NAME = 'columns'

    def __init__(self, grid, user_auth=None, get_params=None):
        """
        @param grid: табличный список
        @type get_params: complex dict
        @param get_params: request.GET
        """
        get_params = get_params or {}
        self._user_auth = user_auth
        self._grid = grid
        self._filter = get_params.get(self.FILTER_PARAM_NAME)
        if self._filter:
            try:
                tree = parse_filter(self._filter)
                assert_columns_exist(tree, grid)
            except FilterError as error:
                log.info('Grid filtering error: "%s"', error)
                self.raise_invalid_filter(self._filter)

        columns_filter_text = get_params.get(self.COLUMNS_PARAM_NAME)
        self.column_names = None
        if columns_filter_text:
            self.column_names = get_column_names(columns_filter_text)
            wrong_columns = set(self.column_names) - set(grid.column_names_by_type())
            if wrong_columns:
                self.raise_invalid_column_id(wrong_columns)

        sort_len = len(self.SORT_KEYWORD)
        self._sort_params = dict(
            (param[sort_len:], direction)
            for param, direction in get_params.items()
            if param.startswith(self.SORT_KEYWORD)
        )

    def raise_invalid_filter(self, filter_string):
        # Translators:
        #  ru: Нельзя фильтровать табличный список по строке "{0}"
        #  en: Cannot filter grid by this string: "{0}"
        raise InvalidDataSentError(_('Cannot filter grid by this string: "{0}"').format(self._filter))

    def raise_invalid_column_id(self, wrong_columns):
        # Translators:
        #  ru: В фильтре столбцов указаны несуществующие столбцы "{0}"
        #  en: Wrong columns specified in columns filter: "{0}"
        raise InvalidDataSentError(
            _('Wrong columns specified in columns filter: "{0}"').format(
                ','.join(sorted([item for item in wrong_columns]))
            )
        )

    def __order_of_keys(self, column_names):
        """
        @rtype: tuple
        """
        order_of_keys = tuple(self._grid.column_names_by_type())
        if column_names:
            # если параметр columns не пустой, то оставить только те столбцы,
            # которые заказывали
            assert set(self.column_names).issubset(set(order_of_keys)), 'Set of columns from GET request is invalid'
            order_of_keys = tuple(self.column_names)
        return order_of_keys

    def _grid_rows(self, hashes=None):
        """
        Вернуть отфильтрованные строки, преобразованные для нового фронтэнда.
        """
        grid = self._grid

        # Данные для некоторых типов колонок, например для колонок данных тикетов Startrek,
        # подгружаются при десериализации. Чтобы иметь возможность фильтровать по ним,
        # десериализуем строки грида перед фильтрацией.
        grid.deserialized_rows = grid.get_rows(self._user_auth, hashes)
        rows = grid.deserialized_rows

        # фильтрация данных
        if self._filter:
            try:
                hashes_in, hashes_out = filter_grid(self._filter, grid, hashes=hashes)
                if hashes:
                    hashes = set(hashes) - set(hashes_out)
                else:
                    hashes = hashes_in
            except FilterError:
                self.raise_invalid_filter(self._filter)

            rows = [row for row in grid.deserialized_rows if row[HASH_KEY] in hashes]

        generator = none_in_cells_gen(
            add_row_id_to_cell_gen(
                text_transformed_to_view_gen(
                    no_wf_version(staff_field(tracker_ticket_field(cut_b64wom1(None, grid), grid), grid)), grid
                ),
                grid,
            ),
            grid,
        )
        generator.send(None)

        for row in rows:
            generator.send(row)

        rows = sort_grid_rows(rows, self.sorting)

        order_of_keys = self.__order_of_keys(self.column_names)

        return rows_for_serialization(rows, order_of_keys)

    @cached_property
    def sorting(self):
        """
        Попытаться взять сортировку из GET-параметров запроса или, из грида.
        """

        all_grid_fields = set(self._grid.column_names_by_type())

        result = []
        for key in self._sort_params:
            if self._sort_params[key] not in self.AVAILABLE_SORTS:
                # You are sorting using "u'ERROR'", use ('desc', 'asc') only; req_id=137626832
                # сортировка некорректная, намекнем, отключив сортировку
                logging.warning(
                    'You are sorting using "%s", use %s only', repr(self._sort_params[key]), repr(self.AVAILABLE_SORTS)
                )
                return []
            if key not in all_grid_fields:
                logging.warning('You cannot sort by "%s", either field never existed or deleted', key)
                # сортировка некорректная, намекнем, отключив сортировку
                return []

            result.append({'name': key, 'type': self._sort_params[key]})

        if not result:
            result = self._grid.access_structure.get('sorting', result)

        return result

    @cached_property
    def rows(self):
        """
        Вернуть отсортированные строки.

        @rtype: list
        """
        return list(self._grid_rows())

    @cached_property
    def structure(self):
        """
        Вернуть структуру грида.

        Возвращает почти точь в точь как в исходном гриде в storage. Но
            * удаляет поля которые не указаны в column_names, если columns заданы
            * если задана фильтрация, заполняет ключ filter_part у каждого поля грида
            * сортировку подменяет на ту, которая указана в sorting (GET-параметрами)

        @rtype: dict
        """
        structure = self._grid.access_structure.copy()

        # hack: разбить поле длины с единицами измерения на два:
        # цифру и строку единиц измерения
        replace_width_in_grid_structure(structure)

        if self._filter:
            filter_parts = get_filter_parts(self._filter)

        for field in structure['fields']:
            for sort in self.sorting:
                if field['name'] == sort['name']:
                    field['sorting'] = sort['type']
                    break
            else:
                if 'sorting' in field:
                    del field['sorting']
            if self._filter:
                if field['name'] in filter_parts:
                    field['filter_part'] = filter_parts[field['name']]

        # Удалить из списка полей те, что не указаны в параметре columns
        if self.column_names:
            chosen_fields = []
            for name in self.column_names:
                for field in structure['fields']:
                    if field['name'] == name:
                        chosen_fields.append(field)
            structure['fields'] = chosen_fields

        if 'sorting' not in structure:
            # у некоторых гридов, созданных в nodejs-бете за время
            # с февраля 2014 по май 2014, отсутствует поле sorting.
            # имя надо проставить поле sorting, если таких гридов
            # будет выявлено много
            # it's okay
            structure['sorting'] = []

        return structure

    def __getattr__(self, item):
        return getattr(self._grid, item)


def rows_for_serialization(rows, order_of_keys):
    """
    Подготовить строки из грида к сериализации.
    Упорядочить по order_of_keys.

    @type rows: list
    @type order_of_keys: tuple
    @rtype: tuple
    @return: генератор туплей
    """
    for row in rows:
        new_row = []
        for cell_key in order_of_keys:
            # KeyError не возможен, потому что целостность грида поддерживается кодом.
            try:
                value = row[cell_key]
            except KeyError as err:
                raise InvalidDataSentError('Unknown column key: {}. Please, contact support.'.format(err))

            if value is None:
                value = {}

            value[HASH_KEY] = cell_key

            new_row.append(value)

        yield tuple(new_row)
