"""
Сериализаторы прав доступа
"""

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki import access as wiki_access
from wiki.api_core.utils import API_ACCESS_TYPES
from wiki.api_frontend.serializers.user_identity import UserIdentityListField
from wiki.api_frontend.serializers.users import SimpleGroupSerializer, UserSerializer
from wiki.pages.logic import access as access_logic

INHERITED_TYPE = API_ACCESS_TYPES.get_key(wiki_access.TYPES.INHERITED)
RESTRICTED_TYPE = API_ACCESS_TYPES.get_key(wiki_access.TYPES.RESTRICTED)


class AccessData(object):
    def __init__(self, page):
        access = access_logic.get_access(page)

        self.type = access.type
        if access.parent_access:
            self.parent_access = access.parent_access.effective_type
        else:
            self.parent_access = None
        if settings.IS_INTRANET:
            self.opened_to_external_flag = access.is_opened_to_external
        self.owner = page.owner
        self.authors = page.get_authors()
        self.restrictions = access.restrictions


class EntitiesWithAccessSerializer(serializers.Serializer):
    """
    Сериализатор для сущностей, имеющий доступ к странице.

    Групп, пользователей и роботных пользователей
    """

    users = UserSerializer(many=True)
    groups = SimpleGroupSerializer(many=True)


class PageAccessSerializer(serializers.Serializer):
    """
    Сериализатор прав доступа к странице.
    Используется в ручке просмотра страницы.
    """

    type = serializers.ChoiceField(choices=API_ACCESS_TYPES.choices(), read_only=True)
    parent_access = serializers.ChoiceField(
        choices=[choice for choice in API_ACCESS_TYPES.choices() if choice[0] != API_ACCESS_TYPES.inherited],
        allow_blank=True,
    )
    if settings.IS_INTRANET:
        opened_to_external_flag = serializers.BooleanField()

    def get_attribute(self, instance):
        """
        Передаем в сериализатор не сам объект, а его обертку.

        @type instance: Page
        """
        return AccessData(instance) if instance else instance


class DetailedAccessSerializer(serializers.Serializer):
    """
    Сериализатор прав доступа к странице с картой наследований
    детей и владельцем страницы
    """

    type = serializers.CharField()
    parent_access = serializers.CharField()
    if settings.IS_INTRANET:
        opened_to_external_flag = serializers.BooleanField()
    owner = UserSerializer()
    authors = UserSerializer(many=True)
    restrictions = EntitiesWithAccessSerializer()


class DetailedAccessSerializerForPOST(DetailedAccessSerializer):
    rejects = EntitiesWithAccessSerializer()
    request_role_url = serializers.CharField()


class AccessModificationSerializer(serializers.Serializer):
    """
    Сериализатор валидирует запрос изменения прав доступа.

    Было бы удобно, если бы rest_framework умел обращаться с полями users, groups, departments
    как умеет обращаться джанго - чтобы туда передавать id сущностей, а rest_framework бросал бы ошибку
    если
    """

    type = serializers.ChoiceField(choices=API_ACCESS_TYPES.choices())
    users = UserIdentityListField(required=False)
    if settings.IS_INTRANET:
        departments = serializers.ListField(child=serializers.IntegerField(min_value=0), required=False)
        groups = serializers.ListField(child=serializers.IntegerField(min_value=0), required=False)
    elif settings.IS_BUSINESS:
        departments = serializers.ListField(child=serializers.CharField(min_length=1), required=False)
        groups = serializers.ListField(child=serializers.CharField(min_length=1), required=False)
    else:
        groups = serializers.ListField(child=serializers.IntegerField(min_value=0), required=False)

    # одно из этих полей обязательно должно быть заполнено
    RESTRICTED_FIELDS = ('users', 'groups', 'departments')

    default_error_messages = {
        # Translators:
        #  ru: При использовании типа "ограниченный" укажите группу или пользователя
        #  en: When using type "restricted", specify any group or user
        'group_or_user_required': _('When using type "restricted", specify any group or user')
    }

    def validate(self, attrs):
        if attrs['type'] == RESTRICTED_TYPE:
            if not any(attrs.get(name) for name in self.RESTRICTED_FIELDS):
                self.fail('group_or_user_required')
        return attrs
