from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_core.serializers import DateTimeTzAwareField
from wiki.api_frontend.serializers.users import UserSerializer
from wiki.pages.models.consts import ACTUALITY_STATUS


class ActualityStatusSerializer(serializers.Field):
    """
    Сериализатор статуса актуальности страницы.
    """

    def to_representation(self, obj):
        return ACTUALITY_STATUS[obj]


class ActualityMarkSerializer(serializers.Serializer):
    status = ActualityStatusSerializer()
    marked_at = DateTimeTzAwareField()
    user = UserSerializer()
    comment = serializers.ReadOnlyField()
    links = serializers.ListField(child=serializers.CharField(min_length=1, trim_whitespace=True))


class ActualityMarkDeserializer(serializers.Serializer):
    actual = serializers.BooleanField()
    comment = serializers.CharField(required=False, min_length=1, trim_whitespace=True, allow_blank=True)
    links = serializers.ListField(child=serializers.CharField(min_length=1, trim_whitespace=True), required=False)

    default_error_messages = {
        # Translators:
        #  ru: Ссылки нельзя указывать для актуальных страниц
        #  en: Links cannot be specified for actual page
        'no_links_for_actual_page': _('Links cannot be specified for actual page'),
    }

    def validate(self, attrs):
        links = attrs.get('links')
        if links is None:
            return attrs
        if attrs['actual'] and links:
            self.fail('no_links_for_actual_page')
        return attrs
