from rest_framework import serializers

from wiki.api_core.serializers.users import UserListItemSerializer
from wiki.api_core.utils import API_USER_ACCESS
from wiki.api_frontend.serializers.base import EntityArraySerializer
from wiki.pages.access import get_bulk_access_status
from wiki.pages.cluster import Cluster, filter_public_page
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.serializer_contexts import UserContext


class AutocompletePageArraySerializer(ContextBoundSerializer[UserContext], EntityArraySerializer):
    """
    Сериализатор ответов автокомплита по супертэгу страницы
    """

    def get_data_array(self, cluster_name):
        """
        вернуть список супертэгов похожих на переданную строчку
        """
        depth = cluster_name.count('/')  # get only same level supertags

        # получим все подходящие для автокомплита страницы
        cluster = Cluster(self.get_context().user)
        pages = cluster.list(cluster_name, depth)

        # получим уровни доступа для каждой подходящей страницы
        access_list = get_bulk_access_status([p.supertag for p in pages], self.get_context().user)

        # уберем из результата те, к которым у пользователя нет доступа
        filter_public = filter_public_page(cluster_name, access_list)

        result = []
        for page in filter(filter_public, pages):
            result.append(
                {
                    'supertag': page.supertag,
                    'tag': page.tag,
                    'url': page.url,
                    'access': access_list.get(page.supertag, None),
                }
            )

        return result

    def to_native(self, template):
        all_supertags = self.get_data_array(template)
        result_supertags = all_supertags[self.start : self.stop]

        self._interpret_access_levels(result_supertags)
        return {'data': result_supertags, 'total': len(all_supertags)}

    def _interpret_access_levels(self, urls_struct):
        """
        Перевести код уровня доступа пользователя к ссылке в термины API

        @type urls_struct: list
        @param urls_struct: список данных о подходящих для автокомплита адресах
        @return: None
        """
        for data in urls_struct:
            data['user_access'] = API_USER_ACCESS.get_name(data['access'])
            del data['access']


class AuthAutocompleteDeserializer(serializers.Serializer):
    q = serializers.CharField(required=True, min_length=1, trim_whitespace=True, allow_blank=False)
    filter = serializers.ChoiceField(choices=('user', 'group'), required=False, default=None)


class AuthAutocompleteUserSerializer(UserListItemSerializer):
    id = serializers.IntegerField()


class AuthAutocompleteGroupSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    name = serializers.CharField()


class AuthAutocompleteSerializer(serializers.Serializer):
    users = AuthAutocompleteUserSerializer(many=True)
    groups = AuthAutocompleteGroupSerializer(many=True)
