"""
Базовые сериализаторы, для наследования
"""

from rest_framework import serializers


class EntityArraySerializer(serializers.Serializer):
    """
    Сериализатор списка объектов, принимает в __init__ start и limit,
    позволяющие контролировать сдвиг и длину в сериализуемом списке.
    """

    def __init__(self, *args, **kwargs):
        """
        Здесь нельзя явно указать start и limit как именованные параметры, т.к. начинаются проблемы с пробросом
        остальных параметров, переданных без указания имени, в родительский класс.

        @param start: позиция элемента в сериализуемом списке, начиная с которого будет идти сериализация.
                      По умолчанию - 0
        @param limit: сколько всего элементов сериализуемого списка возвращать. По умолчанию - 250.
        """
        self.start = kwargs.pop('start', 0) or 0  # если kwargs['start'] == None, все равно нужен 0
        limit = kwargs.pop('limit', 250) or 250  # если kwargs['limit'] == None, все равно нужен 250

        self.stop = self.start + limit

        super().__init__(*args, **kwargs)

    def to_representation(self, instance):
        return self.to_native(instance)

    def to_native(self, obj):
        return {'data': self.get_data_array(obj), 'total': self.get_data_count(obj)}

    def get_data_array(self, obj):
        raise NotImplementedError

    def get_data_count(self, obj):
        raise NotImplementedError


class FakeSerializer(serializers.Serializer):
    """
    Сериализатор-заглушка, используется для мест, где авто-документация не работает
    """

    pass


class NamedIdentifiableSerializer(serializers.Serializer):
    """
    Сериализатор для пары <числовой id, строковое название>.
    """

    id = serializers.IntegerField()
    name = serializers.CharField()


class StringNamedIdentifiableSerializer(serializers.Serializer):
    """
    Сериализатор для пары <строковый id, строковое название>.
    """

    id = serializers.CharField()
    name = serializers.CharField()
