from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_core.serializers import DateTimeTzAwareField, UserSerializer
from wiki.pages.models.comment import Comment
from wiki.pages.models.page import Page
from wiki.pages.models.consts import COMMENTS_STATUS
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.rest_framework import fields
from wiki.utils.serializer_contexts import RequestContext

try:
    from ujson import dumps, loads  # noqa
except ImportError:
    from json import loads, dumps  # noqa


class CommentSerializer(ContextBoundSerializer[RequestContext], serializers.Serializer):
    default_error_messages = {
        'wrong_parent_page': _('Comment {parent.id} belongs to another page'),
    }

    def __init__(self, *args, **kwargs):
        self.page = kwargs.pop('page', None)
        super().__init__(*args, **kwargs)

    id = serializers.ReadOnlyField()
    body = serializers.CharField()
    parent = fields.ModelChoiceField(
        queryset=Comment.objects.filter(status=True),
        required=False,
        allow_null=True,
    )
    user = UserSerializer(read_only=True)
    created_at = DateTimeTzAwareField(read_only=True)
    is_deleted = serializers.SerializerMethodField()

    def get_is_deleted(self, comment):
        return not comment.status

    def validate_parent(self, parent):
        if not parent or not self.page:
            return parent
        if parent.page_id != self.page.id:
            self.fail('wrong_parent_page', parent=parent)
        return parent


class CommentsStatusSerializer(serializers.Serializer):
    """
    Сериализатор статуса доступности комментариев на странице.
    """

    comments_status = serializers.SerializerMethodField()
    # количество страниц в кластере, где одним из авторов является один из авторов корневой страницы
    subpages_count = serializers.SerializerMethodField()

    def get_comments_status(self, page):
        return COMMENTS_STATUS[page.comments_status]

    def get_subpages_count(self, page):
        return Page.active.get_descendants(page.supertag).distinct().count()


class CommentsStatusDeserializer(serializers.Serializer):
    disabled = serializers.BooleanField()
    for_cluster = serializers.BooleanField()
