import logging

from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_core.serializers import DateTimeTzAwareField
from wiki.favorites_v2.models import Bookmark, Folder

logger = logging.getLogger(__name__)


class FolderInputParamsSerializer(serializers.Serializer):
    """
    Сериализатор для папки закладок пользователя.
    """

    name = serializers.CharField(max_length=255)

    default_error_messages = {
        # Translators:
        #  ru: "{name}" зарезервировано для специальной папки
        #  en: "{name}" is reserved for special Folder
        'reserved_for_special': _('"{name}" is reserved for special Folder')
    }

    def validate_name(self, value):
        if value in Folder.RESERVED_FOLDER_NAMES_LIST:
            # нельзя создать папку, имя которой совпадает с одним из зарезервированных имен
            self.fail('reserved_for_special', name=value)


class FolderSerializer(serializers.Serializer):
    name = serializers.CharField(max_length=255)
    favorites_count = serializers.IntegerField()
    type = serializers.CharField(max_length=50)


class FolderModelSerializer(serializers.ModelSerializer):
    class Meta:
        model = Folder
        fields = ('name', 'favorites_count', 'type')


class BookmarkInputSerializer(serializers.Serializer):
    title = serializers.CharField(max_length=500)
    url = serializers.CharField(max_length=1000)
    page_modified_at = DateTimeTzAwareField(required=False)
    folder_name = serializers.CharField(max_length=255)


class BookmarkEditionInputSerializer(serializers.Serializer):
    title = serializers.CharField(max_length=500)


class BookmarkMoveSerializer(serializers.Serializer):
    target_folder_name = serializers.CharField(min_length=1)


class BookmarkSerializer(serializers.Serializer):
    """
    Сериализатор для закладки пользователя.
    """
    id = serializers.IntegerField()
    title = serializers.SerializerMethodField()
    url = serializers.SerializerMethodField()
    page_modified_at = serializers.SerializerMethodField()
    page_last_editor = serializers.SerializerMethodField()

    def get_title(self, bookmark):
        return bookmark.page.title

    def get_url(self, bookmark):
        return bookmark.page.absolute_url

    def get_page_modified_at(self, bookmark):
        return bookmark.page.modified_at

    def get_page_last_editor(self, bookmark):
        return bookmark.page.last_author.username


class BookmarkModelSerializer(serializers.ModelSerializer):
    """
    Сериализатор для закладки пользователя по модели.
    """

    title = serializers.CharField(max_length=500)
    url = serializers.CharField(max_length=1000)
    page_modified_at = DateTimeTzAwareField(required=False)

    class Meta:
        model = Bookmark
        fields = ('id', 'title', 'url', 'page_modified_at', 'page_last_editor')


class BookmarkListModelSerializer(serializers.Serializer):
    """
    Сериализатор списка закладок, сгруппированных по имени папки.
    """

    folder_name = serializers.CharField()
    bookmarks = serializers.ListField(child=BookmarkModelSerializer())


class BookmarkListSerializer(serializers.Serializer):
    """
    Сериализатор списка закладок, сгруппированных по имени папки.
    """

    folder_name = serializers.CharField()
    bookmarks = serializers.ListField(child=BookmarkSerializer())


class LinkedBookmarksDeletionSerializer(serializers.Serializer):
    supertag = serializers.CharField(min_length=1)


class FolderRenameSerializer(serializers.Serializer):
    new_name = serializers.CharField(min_length=1)


class FolderMoveSerializer(serializers.Serializer):
    after = serializers.CharField(min_length=1)
