"""
Сериализаторы файлов
"""
import logging

from rest_framework import serializers
from wiki.api_core.utils import is_docviewer_previewable
from wiki.api_frontend.serializers.base import EntityArraySerializer
from wiki.pages import dao as pages_dao
from wiki.pages.access import is_admin
from wiki.sync.connect.org_ctx import get_org_dir_id
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.docviewer import docviewer_preview_link
from wiki.utils.serializer_contexts import UserContext
from wiki.utils.timezone import make_naive_current

logger = logging.getLogger(__name__)


class FilesArraySerializer(ContextBoundSerializer[UserContext], EntityArraySerializer):
    """
    Сериализатор для списка файлов
    """

    def __init__(self, *args, **kwargs):
        """
        Здесь нельзя явно указать files_for_view как именованный параметр, т.к. начинаются проблемы с пробросом
        остальных параметров, переданных без указания имени, в родительский класс.

        @param files_for_view: список storage_id файлов, инф-цию про которые необходимо вернуть,
        т.е. если есть не пустой параметр files_for_view, значит запрос только про файлы из списка
        """
        self.files_for_view = kwargs.pop('files_for_view')
        super().__init__(*args, **kwargs)

    def get_data_array(self, page):
        file_set = pages_dao.get_files(page)
        if self.files_for_view:
            file_set = file_set.filter(mds_storage_id__in=self.files_for_view)

        dir_org_id = get_org_dir_id()

        files = []
        for file in file_set.order_by('-created_at')[self.start : self.stop]:
            docviewer_url = docviewer_preview_link(file, dir_org_id) if is_docviewer_previewable(file) else None
            files.append(
                {
                    'name': file.name,
                    'url': file.full_path(page.url, file.url),
                    'docviewer_url': docviewer_url,
                    'size': '%.2f' % (file.size / 1024.0 / 1024),  # in MB
                    'description': file.description,
                    'user_name': file.user.username,
                    'user_cloud_uid': file.user.cloud_uid,
                    # localtime проставлен в TimezoneMiddleware.
                    'upload_date': make_naive_current(file.created_at),
                    'can_delete': self.get_context().user in page.get_authors()
                    or self.get_context().user.id == file.user.id
                    or is_admin(self.get_context().user),
                }
            )

        return files

    def get_data_count(self, page):
        return pages_dao.get_files(page).count()


class AttachFileSerializer(serializers.Serializer):
    files = serializers.ListField(child=serializers.CharField(min_length=1))
