
import itertools

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from wiki.legacy.choices import Choices
from rest_framework import serializers

from wiki.api_frontend.serializers.grids.change_grid.base import (
    MULTIPLE_SELECT_TYPE,
    MULTIPLE_STAFF_TYPE,
    TYPES_WITH_OPTIONS,
    StructureChanger,
)
from wiki.api_frontend.serializers.grids.change_grid.default_field_values import for_anything_else, type_map
from wiki.api_frontend.serializers.grids.change_grid.select_column import GridSelectColumnChangesSerializer
from wiki.api_frontend.serializers.grids.errors import MaximumColumnsCountExceeded, TicketDependentFieldWithoutTicket
from wiki.grids.logic.tickets import ticket_dependent_types
from wiki.grids.utils import the_field_types, ticket_field_names
from wiki.grids.utils.base import CHECKBOX_TYPE, SELECT_TYPE, STAFF_TYPE, TICKET_TYPE

ticket_field_choices = list(
    itertools.chain(list(the_field_types.keys()), ticket_dependent_types, (MULTIPLE_SELECT_TYPE, MULTIPLE_STAFF_TYPE))
)


def generate_name(names):
    """
    Вернуть новое имя для столбца.

    @type names: list
    """
    # WIKI-6314: порядковые номера, начиная с 100
    # Поскольку у гридов не бывает очень много столбцов, забиваем на то, что сложность такого поиска свободного id —
    # O(n^2). Зато id будут идти по порядку.
    for candidate in map(str, range(100, 99999)):
        if candidate not in names:
            return candidate


class BaseGridColumnOperation(StructureChanger, serializers.Serializer):
    """
    Предок операций создания и изменения столбцов.

    Все типы полей здесь - не обязательные, потому что наследник
    GridColumnEdited позволяет модифицировать поля столбца по одному.
    """

    WIDTH_UNITS = Choices(pixel=('pixel', 'px'), percent=('percent', '%'))

    default_error_messages = {
        # Translators:
        #  ru: При использовании типа "{0}" укажите список опций
        #  en: When using type "{0}", specify "options" as a list
        'options_required': _('When using type "{0}", specify "options" as a list'),
        'width_without_units': '"width" ({0}) cannot come without "width_units" ({1})',
        'markdown_for_checkbox': 'Specify "markdone" when using field type "checkbox"',
    }

    title = serializers.CharField(required=False, allow_blank=True, default='')
    width = serializers.IntegerField(min_value=0, max_value=100000, required=False, default=None)
    width_units = serializers.ChoiceField(choices=WIDTH_UNITS.choices(), required=False, default=None)
    # используется только в списках и сотрудниках (TYPES_WITH_OPTIONS).
    options = GridSelectColumnChangesSerializer(required=False, default=None)
    # используется только для полей типа checkbox
    markdone = serializers.BooleanField(required=False, default=False)
    # тут есть ошибка в rest_framework - BooleanField нельзя сделать required,
    # оно всегда заполняется default значением, если не передано пользователем.
    # /usr/lib/python2.6/dist-packages/rest_framework/fields.py:296
    required = serializers.BooleanField(required=False, default=False)
    # используется для задания формата отображения имени сотрудника
    format = serializers.CharField(required=False, default='')

    def width_to_string(self, width, width_units):
        """
        @type width: int
        @type width_units: str
        """
        return '{width}{units}'.format(
            width=width,
            units=self.WIDTH_UNITS[width_units],
        )

    def validate(self, attrs):
        """

        @type attrs: dict
        """
        # нельзя указать ширину, не указав единицы.
        width = attrs.get('width')
        units = attrs.get('width_units')
        if width is not None and units is None:  # XOR
            self.fail('width_without_units', width, units)

        # нельзя указать тип select, не указав options
        type = attrs.get('type')
        options = attrs.get('options')
        if type in TYPES_WITH_OPTIONS:
            if options is None:
                self.fail('options_required', type)

        # нельзя указать тип "чекбокс" не указав "markdone"
        if type == CHECKBOX_TYPE and 'markdone' not in attrs:
            self.fail('markdown_for_checkbox')

        return attrs


class GridColumnAdded(BaseGridColumnOperation):
    type = serializers.ChoiceField(choices=[(choice, choice) for choice in ticket_field_choices])

    def validate_type(self, value):
        if not settings.ISSUE_TRACKERS_ENABLED and value in ticket_dependent_types:
            raise serializers.ValidationError(
                ('There are no trackers in this installation, cannot"' '" use field type "{0}"').format(value)
            )
        return value

    def find_merge_conflicts(
        self, grid, user_version, type, title, required=False, width=None, width_units=None, options=None
    ):
        """

        @raise ApplyChangesError
        """
        if settings.ISSUE_TRACKERS_ENABLED:
            if type in ticket_dependent_types:
                if not list(grid.columns_by_type(TICKET_TYPE)):
                    raise TicketDependentFieldWithoutTicket()

    def apply_change(
        self,
        user,
        server_grid,
        user_grid,
        type,
        title,
        required=False,
        width=None,
        width_units=None,
        options=None,
        markdone=None,
        format='',
        **kwargs
    ):
        """
        Добавить столбец.

        @type server_grid: Grid
        @type user_grid: Grid
        """
        limit = settings.LIMIT__GRID_COLS_COUNT
        if limit and len(user_grid.columns_meta()) + 1 > limit:
            raise MaximumColumnsCountExceeded

        self.find_merge_conflicts(server_grid, user_grid, type, title, width, width_units, required, options)

        struct = server_grid.access_structure.copy()

        field = {
            'name': generate_name([field['name'] for field in struct['fields']]),
            'title': title,
            'required': required,
        }
        if width_units:
            field['width'] = self.width_to_string(width, width_units)

        # поля типа MULTIPLE_STAFF, MULTIPLE_CHOICE
        true_type = type
        if type == MULTIPLE_STAFF_TYPE:
            true_type = STAFF_TYPE
            field['multiple'] = True
        elif type == MULTIPLE_SELECT_TYPE:
            true_type = SELECT_TYPE
            field['multiple'] = True

        if true_type == CHECKBOX_TYPE:
            field['markdone'] = markdone

        if true_type in TYPES_WITH_OPTIONS:
            option_serializer = GridSelectColumnChangesSerializer(data=options)
            option_serializer.save(server_grid, field, [])

        if true_type in (STAFF_TYPE, ticket_field_names.assignee, ticket_field_names.reporter):
            field['format'] = format or 'i_first_name i_last_name'

        field['type'] = true_type
        struct['fields'].append(field)
        server_grid.access_structure = struct

        # скопировать (!)
        data = list(server_grid.access_data)

        for row in data:
            row[field['name']] = type_map.get(true_type) or for_anything_else()

        server_grid.access_data = data

        return server_grid
