
from django.conf import settings
from rest_framework import serializers

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_frontend.serializers.grids.change_grid.base import ChangeRowSerializer, LamportClock
from wiki.api_frontend.serializers.grids.errors import MaximumRowsCountExceeded, NoMoveTargetRowsError, NoSuchRowError
from wiki.grids.utils import RowValidationError, dummy_request_for_grids, row_change_action
from wiki.grids.utils.rows import insert_rows_at_idx
from wiki.notifications.models import PageEvent


class GridAddedRow(LamportClock, ChangeRowSerializer):
    """
    Добавление строки в табличный список.

    after_id = -1 означает добавить первой строкой,
    before_id = -1 означает добавить последней строкой
    """

    after_id = serializers.CharField(max_length=20)
    before_id = serializers.CharField(max_length=20, required=False, default='-1')
    data = serializers.DictField()

    new_row_hash = None

    def apply_change(self, user, server_grid, user_grid, after_id, before_id, data=None):
        """
        Добавить столбец из data.

        @type server_grid: Grid
        @type user_grid: Grid
        @type after_id: str
        @type before_id: str
        @type data: dict
        """
        limit = settings.LIMIT__GRID_ROWS_COUNT
        if limit and len(user_grid.access_data) + 1 > limit:
            raise MaximumRowsCountExceeded

        self.check_row_new_data(server_grid, user_grid, data)

        if after_id == 'last':
            if before_id != '-1':
                raise InvalidDataSentError(debug_message='invalid after/before ids pair')
        else:
            # просто формальная проверка наличия строк в пользовательской версии
            # (если всё работает штатно, они всегда там есть)
            for id_to_check in (after_id, before_id):
                if id_to_check == '-1':
                    continue
                if id_to_check not in user_grid.access_idx:
                    # в документе пользователя этой строки не существуют
                    raise NoSuchRowError(debug_message='does not exist in YOUR version')

        if after_id == '-1':
            # строку добавляем в начало
            at_idx = 0
        else:
            at_idx = server_grid.access_idx.get(after_id)
            if at_idx is None:
                # нет уже строки, "после которой"
                if before_id == '-1':
                    # строку добавляем в конец
                    at_idx = len(server_grid.access_data)
                else:
                    before_idx = server_grid.access_idx.get(before_id)
                    if before_idx is None:
                        # и строки "перед которой" тоже уже нет
                        raise NoMoveTargetRowsError
                    at_idx = before_idx - 1
            else:
                at_idx += 1

        try:
            request = dummy_request_for_grids(user)
            (new_row_hash,) = insert_rows_at_idx(server_grid, [data], request=request, row_number=at_idx)
        except RowValidationError as e:
            return self.raise_grid_row_edition_error(server_grid, e)

        self.new_row_hash = new_row_hash
        return server_grid

    def increment_version(self, user, grid):
        # неявно увеличивает счетчик.
        row_change_action[PageEvent.EVENT_TYPES.create](
            grid.access_data[grid.access_idx[self.new_row_hash]], user, grid
        )
