"""
Вся эта структура спроектирована с учетом обратной совместимости со старыми хендлерами.

После отказа от старых хендлеров можно увеличивать значение счетчика более явно.
"""

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_frontend.logic.errors import simplify_grid_row_errors
from wiki.api_frontend.serializers.grids.errors import MaximumCellSizeExceeded, NoSuchColumnError, NoSuchOptionError
from wiki.grids.models import Revision
from wiki.grids.utils import changes_of_structure
from wiki.grids.utils.base import SELECT_TYPE
from wiki.users.core import get_support_contact

MULTIPLE_STAFF_TYPE = 'multiple_staff'
MULTIPLE_SELECT_TYPE = 'multiple_select'

TYPES_WITH_OPTIONS = (SELECT_TYPE, MULTIPLE_SELECT_TYPE)


class LamportClock(object):
    """
    Обязательно увеличить счетчик часов после работы.

    Счетчиком является id последней по времени ревизии данного грида.
    """

    def update(self, instance, validated_data):
        """
        Перегруженный из serializers.Serializer.update.
        """
        user = validated_data.pop('user')
        state_on_server = instance
        state_at_user = validated_data.pop('user_version')
        return self.apply_change(user, state_on_server, state_at_user, **validated_data)

    def apply_change(self, user, state2, state1, **data):
        """
        Применить изменения data на табличном списке state2, используя state1.

        Логика применения такая: переводим из state2 -> state3.

        TODO: убрать state1 из этого метода, унести валидацию в отдельный метод и передавать туда state1, state2.
        @type user: User
        @type state1: Revision
        @type state2: Grid
        @rtype: Grid
        """
        raise NotImplementedError(
            'You should implement this in "{cls}"'.format(
                cls=self.__class__.__name__,
            )
        )

    def pre_perform(self, grid):
        """
        Произвести артефакты перед работой
        """
        pass

    def increment_version(self, user, grid):
        """
        Увеличить счетчик.

        @type changer: Serializer
        """
        raise NotImplementedError()

    def raise_bad_data_from_client(self, non_localized_message=None):
        """
        Отправить человекочитаемую ошибку на клиент о плохом запросе.

        В гридах сложные структуры данных передаются с фронтэнда и с бекэнда.
        Некоторые ошибки которые могут возникнуть теоретически фронтэнд никогда не отправит:
        несуществующий столбец, несуществующая ячейка.
        Делать для них человекочитаемые переводы дорого и нет смысла.
        Бекэнд логгирует ошибку и отправляет 409 на клиент.
        """
        # Translators:
        #  ru: Сообщите %(email_contact)s об этой ошибке
        #  en: Report %(email_contact)s about this error
        msg = _('Report %(email_contact)s about this error') % {'email_contact': get_support_contact()}
        if non_localized_message:
            msg += ' ' + non_localized_message
        raise InvalidDataSentError(msg)

    def raise_grid_row_edition_error(self, grid, row_validation_error):
        """
        @type row_validation_error: RowValidationError
        """
        # поскольку на фронтэнде всегда происходит редактирование только одной ячейки за раз,
        # то мы выводим ошибку так, чтобы фронтэнд не сопоставлял id ячейки ключу словаря errors.
        # можно вывести все ошибки как non_field_messages
        raise InvalidDataSentError(non_field_messages=simplify_grid_row_errors(row_validation_error.errors, grid=grid))


class StructureChanger(LamportClock):
    """
    Признак того, что меняется структура.
    """

    def pre_perform(self, grid):
        self.previous_structure = grid.access_structure.copy()

    def increment_version(self, user, grid):
        # неявно увеличивает счетчик
        changes_of_structure(user, grid, self.previous_structure)


class RowOrderChanger(LamportClock):
    """
    Признак того, что ничего не меняется.
    """

    def increment_version(self, user, grid):
        Revision.objects.create_from_page(grid)


class ChangeRowSerializer(serializers.Serializer):
    @staticmethod
    def check_row_new_data(server_grid, user_grid, data):
        for key in data.keys():
            user_column = user_grid.column_by_name(key)
            server_column = server_grid.column_by_name(key)
            if user_column is None or server_column is None:
                raise NoSuchColumnError

            if isinstance(data[key], str) or isinstance(data[key], str):
                limit = settings.LIMIT__WIKI_TEXT_FOR_GRID_CELL__BYTES
                if data[key] and limit and len(data[key].encode('utf-8')) > limit:
                    raise MaximumCellSizeExceeded(limit)

            if server_column['type'] in TYPES_WITH_OPTIONS:
                selected_options = data[key]
                if not isinstance(selected_options, list):
                    selected_options = [selected_options]
                server_options = list(server_column.get('options'))
                # WIKI-11060: позволяем сохранять пустое значение вместо элемента списка
                server_options.append('')
                if any(option not in server_options for option in selected_options):
                    raise NoSuchOptionError
