import logging
from collections.abc import Iterable

from django.conf import settings
from rest_framework import serializers

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_frontend.serializers.grids.change_grid.add_column import GridColumnAdded
from wiki.api_frontend.serializers.grids.change_grid.add_row import GridAddedRow
from wiki.api_frontend.serializers.grids.change_grid.change_sorting import GridSortingChanged
from wiki.api_frontend.serializers.grids.change_grid.change_title import GridTitleChanged
from wiki.api_frontend.serializers.grids.change_grid.edit_column import GridColumnEdited
from wiki.api_frontend.serializers.grids.change_grid.edit_row import GridEditedRow
from wiki.api_frontend.serializers.grids.change_grid.move_column import GridColumnMoved
from wiki.api_frontend.serializers.grids.change_grid.move_row import GridRowMoved
from wiki.api_frontend.serializers.grids.change_grid.remove_column import GridColumnRemoved
from wiki.api_frontend.serializers.grids.change_grid.remove_row import GridRowRemoved
from wiki.favorites_v2.logic import page_changed
from wiki.grids.models.revision import Revision, get_version_of_grid
from wiki.utils import timezone
from wiki.utils.db import on_commit
from wiki.utils.tasks import TrackLinksTask

logger = logging.getLogger(__name__)


class Changer(object):
    def field_from_native(self, data, files, field_name, into):
        grid = self.parent.object
        self.parent.object = None
        result = super(Changer, self).field_from_native(data, files, field_name, into)
        self.parent.object = grid
        return result

    def validate(self, attrs):
        """
        @type attrs: dict
        """
        keys = list(attrs.keys())
        if not keys:
            raise serializers.ValidationError('You should not pass empty changes')

        if len(keys) > 1:
            raise serializers.ValidationError('Please, apply only one change at a time')

        return attrs

    def save(self, user, user_version, change_unit):
        """

        @type change_unit: dict
        @type user_version: Grid
        @type user: User
        """
        actual_grid_version = self.instance
        saver_name = list(change_unit.keys() & self.get_fields())[0]  # EBI-1092
        changer = self.get_fields()[saver_name].__class__(self.instance, data=change_unit[saver_name])

        changer.pre_perform(actual_grid_version)

        if change_unit[saver_name] is None:
            raise InvalidDataSentError('No details given in "%s"' % saver_name)

        # этого требует rest_framework, но он уже был проверен,
        # т.к. является частью ModifyingSequenceSerializer
        changer.is_valid()

        # меняем документ
        # кварги будут переданы без изменений в update и create сериализатора.
        changer.save(user=user, user_version=user_version)

        # записываем изменения грида в хранилище. Это нужно чтобы корректно сохранить ревизии.
        actual_grid_version.modified_at = timezone.now()
        actual_grid_version.save()

        # в оригинале надо изменять значение счетчика до изменений, но
        # 1) мы в транзакции
        # 2) обратная совместимость вынуждает делать так
        # 3) нам надо создавать ревизию после того как произошла запись в хранилище
        changer.increment_version(user, actual_grid_version)


class GridChangingUnit(Changer, serializers.Serializer):
    added_row = GridAddedRow(required=False)
    row_moved = GridRowMoved(required=False)
    removed_row = GridRowRemoved(required=False)
    edited_row = GridEditedRow(required=False)
    added_column = GridColumnAdded(required=False)
    edited_column = GridColumnEdited(required=False)
    column_moved = GridColumnMoved(required=False)
    removed_column = GridColumnRemoved(required=False)
    title_changed = GridTitleChanged(required=False)
    sorting_changed = GridSortingChanged(required=False)


class ShortChangingUnit(Changer, serializers.Serializer):
    edited_row = GridEditedRow(required=False)


class ModifyingSequenceSerializer(serializers.Serializer):
    version = serializers.CharField(max_length=20, required=False)
    changes = GridChangingUnit(many=True)

    # WIKI-13268: делаем исключение при проверке количества операций для некоторых роботных пользователей
    EXCLUDED_USER_NAMES = ['robot-timetable', 'robot-ang2', 'robot-yang-anticheat', 'robot-yang', 'robot-yang-rating']

    default_error_messages = {
        'version_required': 'Parameter "version" is required',
    }

    def validate_changes(self, changes):
        if not changes:
            raise serializers.ValidationError('value cannot be empty')
        if not isinstance(changes, Iterable):
            raise serializers.ValidationError('value must be an array')

        request = self.context['request']
        operations_number = len(list(changes))
        if (
            operations_number > settings.MAX_CHANGE_GRID_OPERATIONS_COUNT
            and request.user.username not in self.EXCLUDED_USER_NAMES
        ):
            raise serializers.ValidationError(
                'The maximum allowed number of operations in the request data has been exceeded: allowed max number: '
                '{}, but in request data: {}'.format(settings.MAX_CHANGE_GRID_OPERATIONS_COUNT, operations_number)
            )
        elif operations_number > 1:
            logger.info('Number of some Grid operations in the request data: {}'.format(operations_number))

        return changes

    def validate_version(self, version):
        if not Revision.objects.filter(id=version).exists():
            raise serializers.ValidationError(
                'Invalid version parameter value: revision with id={} does not exist'.format(version)
            )

        return version

    def validate(self, attrs):
        operation_with_required_version_param = [
            'added_row',
            'edited_row',
            'added_column',
            'edited_column',
            'column_moved',
            'title_changed',
            'sorting_changed',
        ]
        version = attrs.get('version')
        changes = attrs.get('changes')

        if not version and any(
            list(operation.keys())[0] in operation_with_required_version_param for operation in changes
        ):
            self.fail('version_required')

        return attrs

    def save(self, author_of_changes, grid, **kwargs):
        """

        @type author_of_changes: django.contrib.auth.User
        """
        version = self.initial_data.get('version')
        if version is None:
            user_grid = None
        else:
            user_grid = get_version_of_grid(version)

        self.merge_changes(grid, user_grid, self.initial_data['changes'], author_of_changes)

        on_commit(lambda: TrackLinksTask().delay(grid.id, is_grid=True, is_new_page=False))
        page_changed(grid, was_created=False)

    def merge_changes(self, state_on_server, state_at_user, changes, user):
        """
        Применить изменения к гриду.

        @type state_on_server: Grid
        @type changes: list
        @type state_at_user: Grid
        @type user: User
        """
        for single_change_operation in changes:
            change_serializer = GridChangingUnit(state_on_server, data=single_change_operation)

            # уже валиден
            change_serializer.save(user, state_at_user, single_change_operation)


class ShortModifyingSequenceSerializer(serializers.Serializer):
    changes = ShortChangingUnit(many=True)
