
import logging

from rest_framework import serializers

from wiki.api_frontend.serializers.grids import errors
from wiki.api_frontend.serializers.grids.change_grid.base import StructureChanger

log = logging.getLogger(__name__)


class GridSortingChanged(StructureChanger, serializers.Serializer):
    sorting = serializers.ListField(child=serializers.DictField(), required=True)

    default_error_messages = {
        'sorting_invalid': 'invalid value',
    }

    def validate_sorting(self, sorting):
        """
        @type attrs: dict
        """
        for column in sorting:
            if type(column) is not dict:
                self.fail('sorting_invalid')
            if len(column) != 1:
                self.fail('sorting_invalid')
            key = list(column.keys())[0]
            value = list(column.values())[0]
            if not isinstance(key, str) or not isinstance(value, str):
                self.fail('sorting_invalid')
            if value != 'asc' and value != 'desc':
                self.fail('sorting_invalid')
        return sorting

    def apply_change(self, user, obj, user_version, sorting, **kwargs):
        """
        Изменить сортировку по-умолчанию.

        @type obj: Grid
        @type user_version: Grid
        @type sorting: list
        """
        self.check_conflicts(actual_version=obj, user_version=user_version)

        struct = obj.access_structure.copy()

        struct_sorting = []
        for column in sorting:
            name = list(column.keys())[0]
            if obj.column_by_name(name) is None:
                # явно ошибка клиента
                log.error('Client sent unknown column "%s"', name)
                self.raise_bad_data_from_client('Client sent unknown column')
            struct_sorting.append({'name': name, 'type': list(column.values())[0]})
        struct['sorting'] = struct_sorting
        obj.access_structure = struct

        return obj

    def check_conflicts(self, actual_version, user_version):
        user_version_sorting = user_version.access_structure['sorting']
        current_sorting = actual_version.access_structure['sorting']

        if user_version_sorting != current_sorting:
            raise errors.SortingHasChangedError(
                {
                    'actual_grid_version': current_sorting,
                    'your_version': user_version_sorting,
                }
            )
