
from rest_framework import serializers

from wiki.api_frontend.serializers.grids.change_grid.add_column import BaseGridColumnOperation
from wiki.api_frontend.serializers.grids.change_grid.base import TYPES_WITH_OPTIONS
from wiki.api_frontend.serializers.grids.change_grid.select_column import GridSelectColumnChangesSerializer
from wiki.api_frontend.serializers.grids.errors import ColumnHasChangedError, NoSuchColumnError
from wiki.grids.logic import columns as columns_logic
from wiki.grids.utils.base import CHECKBOX_TYPE


class GridColumnEdited(BaseGridColumnOperation):
    """
    Нельзя менять тип.
    """

    name = serializers.CharField()

    def find_merge_conflicts(self, grid, user_version, name):
        """
        Найти конфликты мерджа.

        @raise ApplyChangesError
        @rtype: dict
        """

        previous_struct_column = user_version.column_by_name(name)
        if previous_struct_column is None:
            # пользователь указал имя столбца, которого не существует
            # в его версии
            raise NoSuchColumnError

        current_column = grid.column_by_name(name)
        if current_column is None:
            # вероятно, удалили нужный нам столбец
            raise NoSuchColumnError

        var_names = ['title', 'required', 'width', 'markdone']
        for name in var_names:
            if current_column.get(name) == previous_struct_column.get(name):
                continue
            # содержимое поменялось кем-то в промежутке
            # между текущей версией и версией пользователя
            raise ColumnHasChangedError()

        return previous_struct_column

    def apply_change(
        self,
        user,
        server_grid,
        user_grid,
        name,
        title,
        required=False,
        width=None,
        width_units=None,
        options=None,
        markdone=None,
        **kwargs
    ):
        """
        Отредактировать описание столбца.

        @type server_grid: Grid
        @type user_grid: Grid
        """
        user_column = self.find_merge_conflicts(server_grid, user_grid, name)

        struct = server_grid.access_structure.copy()

        server_column = columns_logic.column_by_name_from_access_structure_fields(struct['fields'], name)

        field_type = server_column['type']
        server_column['required'] = required
        server_column['title'] = title

        if width_units:
            server_column['width'] = self.width_to_string(width, width_units)
        else:
            # пользователь хочет задать ширину "по-умолчанию"
            if 'width' in server_column:
                del server_column['width']

        if field_type in TYPES_WITH_OPTIONS:
            option_serializer = GridSelectColumnChangesSerializer(data=options)
            option_serializer.save(server_grid, server_column, user_column.get('options'))

        if field_type == CHECKBOX_TYPE:
            server_column['markdone'] = markdone

        server_grid.access_structure = struct

        # TODO: надо модифицировать данные тут
        return server_grid
