
from copy import deepcopy

from rest_framework import serializers

from wiki.api_frontend.serializers.grids.change_grid.base import ChangeRowSerializer, LamportClock
from wiki.api_frontend.serializers.grids.errors import CellHasChangedError, NoSuchRowError
from wiki.grids.utils import RowValidationError, edit_rows, row_change_action
from wiki.grids.utils import rows as rows_utils
from wiki.grids.utils.base import dummy_request_for_grids
from wiki.notifications.models import PageEvent


class GridEditedRow(LamportClock, ChangeRowSerializer):
    """
    Редактирование строки в табличном списке.

    """

    id = serializers.CharField(max_length=20)
    data = serializers.DictField()

    # айдишник изменяемой строки
    _row_hash = None
    # устаревшее состояние строки
    _previous_state_of_row = None

    def row_hash_to_change_and_conflicts(self, user, server_grid, user_grid, row_id, data):
        """
        Вернуть ключ рядка, который надо изменять и проверить конфликты.

        @raise: NoSuchRowError
        @rtype: str
        """
        self.check_row_new_data(server_grid, user_grid, data)

        try:
            user_idx = user_grid.access_idx[row_id]
            user_row = user_grid.access_data[user_idx]
        except KeyError:
            raise NoSuchRowError(debug_message='does not exist in YOUR version')

        try:
            # либо в новом индексе есть id, либо этот ряд уже не существует.
            idx = server_grid.access_idx[row_id]
        except KeyError:
            # удален
            raise NoSuchRowError

        server_row = server_grid.access_data[idx]
        if rows_utils.are_cells_changed(server_row, user_row, data):
            raise CellHasChangedError

        return row_id

    def apply_change(self, user, obj, user_version, id, data, **kwargs):
        """
        Отредактировать строку.

        @type obj: Grid
        @type user_version: Grid
        @type id: str
        @param id: id строки
        @type data: dict
        """
        self._row_hash = self.row_hash_to_change_and_conflicts(user, obj, user_version, id, data)

        # форматируем от анонимного пользователя, это возможный источник багов.
        request = dummy_request_for_grids(user)
        self._previous_state_of_row = deepcopy(obj.get_rows(request.user_auth, [self._row_hash])[0])
        for key in data:
            if data[key] is False:
                data[key] = ''

        try:
            edit_rows(obj, {self._row_hash: data}, request)
        except RowValidationError as e:
            self.raise_grid_row_edition_error(obj, e)

        return obj

    def increment_version(self, user, grid):
        # неявно увеличивает счетчик.
        row_change_action[PageEvent.EVENT_TYPES.edit](
            grid.get_rows(user_auth=None, hashes=[self._row_hash])[0],
            user,
            grid,
            old_row=self._previous_state_of_row,
        )
