
from collections.abc import Iterable

from django.utils.translation import ugettext_lazy
from rest_framework import serializers

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_frontend.serializers.grids.errors import DuplicateOptionError, NoSuchOptionError


class AddOption(serializers.Serializer):
    """
    Добавить новую опцию в конец списка.
    """

    value = serializers.CharField()

    def update(self, obj, validated_data):
        _, server_column, user_options = obj

        server_options = server_column.setdefault('options', [])

        value = validated_data['value']
        if value in user_options:
            # Дубликаты в списке запрещены.
            raise InvalidDataSentError('Such option already exists in YOUR version')

        if value not in server_options:
            # Дубликаты в списке запрещены, если такое значение уже кто-то добавл, игнорируем.
            server_options.append(value)

        user_options.append(value)

        return obj


class RenameOption(serializers.Serializer):
    """
    Переименовать опцию.
    Переименование отразится в данных грида.
    """

    index = serializers.IntegerField()
    new_value = serializers.CharField()

    def update(self, obj, validated_data):
        server_grid, server_column, user_options = obj

        if user_options is None:
            raise NoSuchOptionError(debug_message='there are no options in YOUR version')

        server_options = server_column.get('options')
        if server_options is None:
            raise NoSuchOptionError

        index = validated_data['index']
        if index >= len(user_options):
            raise NoSuchOptionError(debug_message='no such option in YOUR version')

        old_value = user_options[index]
        try:
            server_index = server_options.index(old_value)
        except ValueError:
            raise NoSuchOptionError

        new_value = validated_data['new_value']
        if new_value in server_options:
            # Дубликаты в списке запрещены.
            raise DuplicateOptionError(new_value)

        server_options[server_index] = new_value
        user_options[index] = new_value

        column_name = server_column['name']
        for row in server_grid.access_data:
            if column_name not in row:
                # Значит мы добавляем колонку, и в данных ее еще нет.
                break
            selected_options = row[column_name]['raw']
            for index, option in enumerate(selected_options):
                if option == old_value:
                    selected_options[index] = new_value
                    # В списке нет дубликатов, так что сразу выходим.
                    break

        return obj


class RemoveOption(serializers.Serializer):
    """
    Удалить опцию.
    Опция будет удалена из данных грида.
    """

    index = serializers.IntegerField()

    def update(self, obj, validated_data):
        server_grid, server_column, user_options = obj

        if user_options is None:
            raise NoSuchOptionError(debug_message='there are no options in YOUR version')

        index = validated_data['index']

        if index >= len(user_options):
            raise NoSuchOptionError(debug_message='no such option in YOUR version')
        value = user_options[index]
        del user_options[index]

        server_options = server_column.get('options')
        if server_options is None:
            # опций вообще нет, в том числе и той, которую нужно удалить, ничего не делаем
            return obj

        try:
            server_index = server_options.index(value)
        except ValueError:
            # уже нет такой опции, ничего не делаем
            return obj

        del server_options[server_index]
        column_name = server_column['name']
        for row in server_grid.access_data:
            if column_name not in row:
                # Значит мы добавляем колонку, и в данных ее еще нет.
                break
            selected_options = row[column_name]['raw']
            for index, option in enumerate(selected_options):
                if option == value:
                    del selected_options[index]
                    # В списке нет дубликатов, так что сразу выходим.
                    break

        return obj


class MoveOption(serializers.Serializer):
    """
    Переместить опцию.
    """

    old_index = serializers.IntegerField()
    new_index = serializers.IntegerField()

    def update(self, obj, validated_data):
        server_grid, server_column, user_options = obj

        if user_options is None:
            raise NoSuchOptionError(debug_message='there are no options in YOUR version')

        server_options = server_column.get('options')
        if server_options is None:
            raise NoSuchOptionError

        old_index = validated_data['old_index']
        new_index = validated_data['new_index']

        if old_index >= len(user_options):
            raise NoSuchOptionError(debug_message='no such option in YOUR version')
        value = user_options[old_index]
        try:
            server_index = server_options.index(value)
        except ValueError:
            raise NoSuchOptionError

        if new_index >= len(user_options):
            raise InvalidDataSentError('New index %d is out of bounds in your version' % new_index)

        user_options.pop(old_index)
        user_options.insert(new_index, value)

        if new_index >= len(server_options):
            new_index = len(server_options)
        server_options.pop(server_index)
        server_options.insert(new_index, value)

        return obj


class Changer(object):
    def validate(self, attrs):
        keys = list(attrs.keys())
        if not keys:
            raise InvalidDataSentError('You should not pass empty changes')

        if len(keys) > 1:
            raise InvalidDataSentError('Please, apply only one change at a time')

        return attrs

    def save(self, server_grid, server_column, user_options, change_unit):
        saver_name = list(change_unit.keys())[0]
        changer = self.get_fields()[saver_name].__class__(
            (server_grid, server_column, user_options), data=change_unit[saver_name]
        )

        if change_unit[saver_name] is None:
            raise InvalidDataSentError('No details given in "%s"' % saver_name)

        if not changer.is_valid():
            raise InvalidDataSentError(changer.errors)

        changer.save()


class SelectColumnChangingUnit(Changer, serializers.Serializer):
    add_option = AddOption(required=False)
    rename_option = RenameOption(required=False)
    remove_option = RemoveOption(required=False)
    move_option = MoveOption(required=False)


class GridSelectColumnChangesSerializer(serializers.Serializer):
    changes = SelectColumnChangingUnit(many=True)

    def validate_changes(self, changes):
        if not isinstance(changes, Iterable):
            raise InvalidDataSentError('Select column changes must be an array')
        return changes

    def save(self, server_grid, server_column, user_options, **kwargs):
        for single_change_operation in self.initial_data['changes']:
            change_serializer = SelectColumnChangingUnit(
                (server_grid, server_column, user_options), data=single_change_operation
            )

            # уже валиден
            change_serializer.save(server_grid, server_column, user_options, single_change_operation)

        # В итоге должен получиться непустой список options.
        if not server_column.get('options'):
            raise InvalidDataSentError(
                # Translators:
                #  ru: Список опций должен быть непустым
                #  en: Option list must be not empty
                ugettext_lazy('grids.Edit:emptySelectOptions')
            )
