

from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.rest_api_error import RestApiError


class ApplyChangesError(RestApiError):
    """
    Базовая ошибка сохранения изменений.

    Не должна быть брошена напрямую, бросайте наследников.
    """

    # текущая версия грида
    actual_version = None
    # версия грида, с которой пришел пользователь
    user_version = None


class ConflictingChangeError(ApplyChangesError):
    """
    Конфликт при попытке смерджить изменения пользовательской версии и версии на бекэнде.

    Пользователь сам должен решить, что делать. Лучше всего пользователю
    получить табличный список с бекэнда заново и попробовать создать операцию
    еще раз, над обновленными данными.

    Старайтесь не кидать это исключение, используйте наследников, это даст больше информации потребителю.
    """

    error_code = 'OPERATION_MAKES_NO_SENSE'
    debug_message = (
        'The grid has probably changed so that desired operation makes no sense'
        ' (or user has intentionally sent such an operation).'
        'This may happen if user has an outdated version of the grid.'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Обновите этот табличный список и повторите действие
        #  en: Reload this grid and repeat your action
        _('Reload this grid please and repeat operation'),
    ]


class ColumnHasChangedError(ConflictingChangeError):
    """
    Столбец так изменился, что над ним нельзя применить операцию.
    """

    error_code = 'COLUMN_HAS_CHANGED'
    detail = 'The column has changed so much, that operation cannot be applied.'
    _non_field_messages = [
        # Translators:
        #  ru: Столбец был изменен, обновите страницу
        #  en: Column has been changed probably, refresh page
        _('Column has been changed, refresh grid'),
    ]


class NoSuchRowError(ConflictingChangeError):
    """
    Запрошена строка, которой не существует.
    """

    error_code = 'NO_SUCH_ROW'

    debug_message = (
        'User has requested operation on a row, which does not exist' ' it may have been deleted or never existed'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Строка вероятно уже удалена, обновите страницу
        #  en: Row has already been deleted probably, refresh page
        _('Row has already been deleted probably, refresh grid'),
    ]


class NoMoveTargetRowsError(ConflictingChangeError):
    """
    Нужно поместить перемещаемую/добавляемую строку после или до строк, которых уже нет обеих.
    """

    error_code = 'NO_TARGET_ROWS'

    debug_message = (
        'User has requested to move after or before rows, which both do not exist.'
        ' They may have been deleted or never existed.'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Строки, возле которых нужно поместить строку, вероятно уже удалены, обновите страницу
        #  en: Probably rows to place the row nearby probably have already been deleted, please refresh the page
        _('Probably rows to place the row nearby have already been deleted, please refresh the page'),
    ]


class TicketDependentFieldWithoutTicket(ConflictingChangeError):
    """
    Чтобы использовать поля, зависящие от тикетов, надо добавить поле типа "тикет".
    """

    error_code = 'CREATE_TICKET_FIELD_FIRST'
    debug_message = 'You should create field of "ticket" type first' ', then a field of dependent type'


class CellHasChangedError(ConflictingChangeError):
    """
    Строка изменилась так, что нельзя смерджить операции.
    """

    error_code = 'ROW_HAS_CHANGED'
    detail = (
        'Cell has changed in that way, this operation cannot be merged'
        'Please, update your version of grid and try again'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Ячейка изменилась, пожалуйста обновите страницу
        #  en: Cell has been changed, please refresh the page
        _('Cell has been changed, please refresh the page'),
    ]


class TitleHasChangedError(ConflictingChangeError):
    error_code = 'TITLE_HAS_CHANGED'
    detail = (
        'Title has changed in that way, this operation cannot be merged'
        'Please, update your version of grid and try again'
    )


class SortingHasChangedError(ConflictingChangeError):
    error_code = 'SORTING_HAS_CHANGED'
    detail = (
        'Sorting has changed in that way, this operation cannot be merged'
        'Please, update your version of grid and try again'
    )


class NoSuchColumnError(ConflictingChangeError):
    """
    Такого столбца не существует.
    """

    error_code = 'NO_SUCH_COLUMN'
    debug_message = (
        'User has requested operation on a column, which does not exist' ' it may have been deleted or never existed'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Столбец вероятно уже удален, обновите страницу
        #  en: Column has already been deleted probably, refresh page
        _('Column has already been deleted probably, refresh grid'),
    ]


class NoSuchOptionError(ConflictingChangeError):
    """
    Указана опция, которой не существует.
    """

    error_code = 'NO_SUCH_OPTION'

    debug_message = (
        'User has requested operation on an option, which does not exist' ' it may have been deleted or never existed'
    )
    _non_field_messages = [
        # Translators:
        #  ru: Такого элемента уже нет в списке, обновите страницу
        #  en: The option is no longer in the list, please refresh the page
        _('The option is no longer in the list, please refresh the grid'),
    ]


class DuplicateOptionError(ConflictingChangeError):
    """
    Указана опция, которой не существует.
    """

    error_code = 'DUPLICATE_OPTION'

    def __init__(self, option_name):
        super(DuplicateOptionError, self).__init__(
            debug_message='User has requested to add an option "%s", which already exists' % option_name,
            non_field_messages=[
                # Translators:
                # ru: Элемент "%s" уже есть в списке, обновите страницу
                #  en: Option "%s" already exists in the list, please refresh the page
                _('Option "%s" already exists in the list, please refresh the grid')
                % option_name,
            ],
        )


class MaximumRowsCountExceeded(ApplyChangesError):
    error_code = 'MAXIMUM_ROWS_COUNT_EXCEEDED'

    def __init__(self):
        super(MaximumRowsCountExceeded, self).__init__(
            debug_message='Maximum grid rows count exceeded',
            non_field_messages=[
                # Translators:
                #  ru: Достигнуто максимальное количество строк
                #  en: Exceeded maximum rows count
                _('Maximum grid rows count exceeded'),
            ],
        )


class MaximumColumnsCountExceeded(ApplyChangesError):
    error_code = 'MAXIMUM_COLUMNS_COUNT_EXCEEDED'

    def __init__(self):
        super(MaximumColumnsCountExceeded, self).__init__(
            debug_message='Maximum grid columns count exceeded',
            non_field_messages=[
                # Translators:
                #  ru: Достигнуто максимальное количество столбцов
                #  en: Exceeded maximum columns count
                _('Maximum grid columns count exceeded'),
            ],
        )


class MaximumCellSizeExceeded(ApplyChangesError):
    error_code = 'MAXIMUM_CELL_SIZE_EXCEEDED'

    def __init__(self, limit):
        limit /= 1 << 10
        super(MaximumCellSizeExceeded, self).__init__(
            debug_message='Maximum grid cell size %d Kb exceeded' % limit,
            non_field_messages=[
                # Translators:
                #  ru: Превышен максимальный размер текста в %d Кб
                #  en: Maximum text size %d Kb exceeded
                _('Maximum grid cell size %d Kb exceeded')
                % limit,
            ],
        )


class ObjectIsLockedForUpdate(ApplyChangesError):
    """
    Не получилось взять лок на объект в базе данных для обновления данных в таблице, так как в настоящий момент
    обрабатывается другой запрос, который поставил лок командой select_for_update().
    """

    error_code = 'OBJECT_IS_LOCKED_FOR_UPDATE'

    debug_message = (
        'At this time, the data change request in the table cannot be processed because '
        'the table is locked by another data change request.'
    )
    _non_field_messages = [
        # Translators:
        #  ru: В настоящий момент запрос не может быть обработан, так как данные в таблице заблокированы другим запросом
        #  en: The request cannot be processed at this time because the data in the table is blocked by another query
        _('The request cannot be processed at this time because the data in the table is blocked by another query'),
    ]
