"""
Сериализаторы гридов (как отдельной сущности)
"""

import logging

from rest_framework import serializers

from wiki.api_core.serializers import DateTimeTzAwareField, UserListItemSerializer, UserSerializer
from wiki.api_frontend.serializers.pages import PageAttributesSerializer

log = logging.getLogger(__name__)


class GridDataSerializer(serializers.Serializer):
    """
    Сериализатор грида
    """

    structure = serializers.ReadOnlyField()
    rows = serializers.ReadOnlyField()
    version = serializers.CharField(source='get_page_version')

    def __init__(self, *args, **kwargs):
        self.only = kwargs.pop('only', None)
        super(GridDataSerializer, self).__init__(*args, **kwargs)

    @property
    def fields(self):
        """
        Отдаем только поля, перечисленные в only, если атрибут не None.
        Если нужно будет в других сериализаторах то же самое сделать — нужно
        будет либо базовый викисериализатор сделать с таким поведением, либо
        миксин.
        """
        all_fields = super(GridDataSerializer, self).fields
        if self.only is None:
            return all_fields
        return {name: field for name, field in list(all_fields.items()) if name in self.only}

    @property
    def data(self):
        return super(GridDataSerializer, self).data


class GridTranslationSerializer(PageAttributesSerializer):
    """
    Сериализатор перевода грида
    """

    structure = serializers.DictField()
    # rows - это список списков (матрица). В каждой ячейке - маппинг.
    rows = serializers.ListField(child=serializers.ListField(child=serializers.DictField()))


class GridRevisionSerializer(GridDataSerializer):
    author = UserSerializer()
    created_at = DateTimeTzAwareField()


class GridConcurrentChangesSerializer(serializers.Serializer):
    editors = serializers.ListField(child=UserListItemSerializer())
