"""
Сериализаторы страниц
"""
import logging
from dataclasses import dataclass
from itertools import chain
import waffle

from django.conf import settings
from django.http import Http404
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers
from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.serializers import DateTimeTzAwareField
from wiki.api_core.serializers.pages import PageTypeSerializer
from wiki.api_core.waffle_switches import OWNER_AS_FIRST_AUTHOR
from wiki.api_frontend.serializers.access import PageAccessSerializer
from wiki.api_frontend.serializers.favorites import BookmarkModelSerializer, BookmarkSerializer
from wiki.api_frontend.serializers.org import OrganizationSerializer
from wiki.api_frontend.serializers.user_identity import UserIdentityListField
from wiki.api_frontend.serializers.users import UserSerializer
from wiki.favorites_v2.models import Bookmark
from wiki.favorites.models import Bookmark as NewBookmark
from wiki.org import get_org
from wiki.pages.api import page_toc
from wiki.pages.constants import ReservedCreationError
from wiki.pages.dao import page as pages_dao
from wiki.pages.dao.revision import get_revision_by_id
from wiki.pages.logic import subscription as logic_subscription
from wiki.pages.logic import tags
from wiki.pages.logic.edit import save_page_from_view
from wiki.pages.logic.files import get_user_css
from wiki.pages.models import Notification, Page
from wiki.pages.models.copy_on_write_page import CopyOnWritePage
from wiki.subscriptions.logic import get_user_page_subscriptions
from wiki.users.dao import get_users_by_identity
from wiki.users.logic.settings import uses_new_favorites, uses_new_subscriptions
from wiki.utils import errors, qr
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.rest_framework.fields import TagField
from wiki.utils.tvm2 import get_user_ticket
from wiki.utils.serializer_contexts import UserContext, RequestContext
from wiki.utils.supertag import tag_to_supertag

try:
    from ujson import dumps, loads  # noqa
except ImportError:
    from json import dumps, loads  # noqa

logger = logging.getLogger(__name__)


class StorageError(Exception):
    pass


class NotifierSerializer(serializers.Serializer):
    id = serializers.ReadOnlyField()
    message = serializers.ReadOnlyField()
    url = serializers.ReadOnlyField()


class BreadcrumbSerializer(serializers.Serializer):
    tag = serializers.CharField()
    url = serializers.CharField()
    title = serializers.CharField()
    is_active = serializers.BooleanField()


class UserCssSerializer(serializers.CharField):
    """
    Находит файлик user.css у этой страницы или родителей.
    """

    def to_representation(self, page):
        return get_user_css(page)


@dataclass
class UserPageContext(UserContext):
    page: Page


class PageAttributesSerializer(ContextBoundSerializer[UserPageContext], serializers.Serializer):
    """
    Cериализатор атрибутов страницы.
    """

    page_type = PageTypeSerializer()
    is_redirect = serializers.BooleanField(source='has_redirect')
    breadcrumbs = BreadcrumbSerializer(many=True)
    owner = UserSerializer()
    authors = serializers.SerializerMethodField()
    last_author = UserSerializer()
    access = PageAccessSerializer()
    current_user_subscription = serializers.SerializerMethodField()
    version = serializers.CharField(source='get_page_version')
    user_css = serializers.SerializerMethodField(required=False, allow_null=True)
    bookmark = serializers.SerializerMethodField()
    is_official = serializers.BooleanField(read_only=True)
    created_at = DateTimeTzAwareField()
    modified_at = DateTimeTzAwareField()
    actuality_status = serializers.SerializerMethodField()
    tag = serializers.ReadOnlyField()
    supertag = serializers.ReadOnlyField()
    url = serializers.ReadOnlyField()
    title = serializers.ReadOnlyField()
    lang = serializers.ReadOnlyField()
    qr_url = serializers.SerializerMethodField()
    with_new_wf = serializers.BooleanField()
    is_readonly = serializers.BooleanField()
    comments_count = serializers.IntegerField(
        source='comments',
        read_only=True,
    )
    comments_status = serializers.SerializerMethodField()
    org = serializers.SerializerMethodField()
    notifier = serializers.SerializerMethodField()

    def get_notifier(self, page):
        notification = Notification.objects.filter(is_active=True)
        notification = notification.exclude(disabled_for_users__in=[self.get_context().user])
        notification = notification.order_by('id').first()

        if notification:
            return NotifierSerializer().to_representation(notification)
        else:
            return {}

    def get_comments_status(self, page):
        from wiki.pages.models.consts import COMMENTS_STATUS

        return COMMENTS_STATUS[page.comments_status]

    def get_actuality_status(self, page):
        from wiki.pages.models.consts import ACTUALITY_STATUS

        return ACTUALITY_STATUS[page.realtime_actuality_status]

    def get_authors(self, page):
        if isinstance(page, CopyOnWritePage):
            authors = page.authors()

        elif waffle.switch_is_active(OWNER_AS_FIRST_AUTHOR) and (owner := page.owner):
            co_authors = page.authors.exclude(id=owner.id).order_by('id')
            authors = [owner] + list(co_authors)
        else:
            authors = page.authors.all()

        return UserSerializer(authors, many=True).data

    def get_bookmark(self, page):
        """
        Вернуть закладку на данную страницу, если она есть в закладках Избранное у пользователя.
        """
        user = self.get_context().user
        if uses_new_favorites(user):
            try:
                bookmark = NewBookmark.objects.get(page=page, user=user, page__org=get_org())
            except NewBookmark.DoesNotExist:
                return None

            return BookmarkSerializer(bookmark).data
        else:
            try:
                return BookmarkModelSerializer(
                    Bookmark.objects.filter(supertag=page.supertag, folder__user=user, folder__org=get_org())[0]
                ).data
            except IndexError:
                # нет такой закладки
                return None

    @staticmethod
    def get_user_css(page):
        return get_user_css(page)

    def get_current_user_subscription(self, value):
        # Информация о подписке на страницу текущего пользователя:
        user, page = self.get_context().user, self.get_context().page
        if not page:
            return 'none'
        if uses_new_subscriptions(user):
            page_watch = get_user_page_subscriptions(user, page)
        else:
            page_watch = logic_subscription.get_user_page_watch(user, page)

        if page_watch:
            if page_watch.is_cluster:
                return 'cluster'
            return 'page'
        return 'none'

    def get_qr_url(self, page):
        return qr.generate_qr_url(page.tag, self.get_context().user)

    def get_org(self, page):
        return OrganizationSerializer(page.org).data if page.org else None


class PageWikiTextSerializer(PageAttributesSerializer):
    """
    Сериализатор содержимого страницы в формате вики-текста.
    """

    body = serializers.SerializerMethodField()

    def get_body(self, page):
        try:
            return page.body  # raises APIError
        except APIError:
            raise StorageError()


class PageTableOfContentSingleSerializer(ContextBoundSerializer[RequestContext], serializers.Serializer):
    """
    Сериализатор оглавления страницы для отдельной ручки
    """

    toc = serializers.SerializerMethodField()

    def get_toc(self, page):
        request = self.get_context().request
        user_auth = request.user_auth
        # User ticket нужен на случай инклюдов на странице,
        # по которым также стоится toc
        user_ticket = None
        if user_auth:
            user_ticket = user_auth.tvm2_user_ticket
            if user_ticket is None:
                user_ticket = get_user_ticket(
                    user_auth.user_ip,
                    user_auth.server_host,
                    user_auth.oauth_token,
                    user_auth.sessionid,
                )
        return page_toc(page, user_ticket)


def who_edited_the_page(revision_id):
    """
    Вернуть ошибку на языке пользователя.
    """
    revision = get_revision_by_id(revision_id)
    # Translators:
    #  ru: Страница была изменена {0} пока вы ее редактировали. Скопируйте ее в буфер обмена и перезагрузите страницу.
    #  en: Page has been changed by {0} while you were editing it. Copy your changes to clipboard and reload the page.
    return revision.author.staff.inflections.ablative


class PageEditionSerializer(serializers.Serializer):
    """
    Валидирует входящий запрос на создание страницы.
    """

    body = serializers.CharField(required=True, allow_blank=True, trim_whitespace=False)
    section_id = serializers.IntegerField(min_value=1, default=None)
    title = serializers.CharField(min_length=1, default=None)
    showpage = TagField(required=False, page_should_exist=True)
    version = serializers.CharField(min_length=1, default=None)
    authors = UserIdentityListField(required=False)
    page_type = serializers.ChoiceField(choices=list(chain(*Page.TYPES.choices())), required=False, default=None)

    default_error_messages = {
        # Translators:
        #  ru: Чтобы создать страницу надо указать title
        #  en: Title is required to create pages
        'title_for_new_pages_required': _('Title is required to create pages'),
        # Translators:
        #  ru: Нельзя редактировать секцию страницы, которой еще не существует
        #  en: Cannot edit section of page that does not exist
        'no_sections_in_new_page': _('Cannot edit section of page that does not exist'),
        # Translators:
        #  ru: Нельзя редактировать табличный список как страницу
        #  en: Cannot edit grid as a page
        'editing_grid_as_page': _('Cannot edit grid as a page'),
        # Translators:
        # ru: Страница была изменена {author} пока вы ее редактировали.
        # Скопируйте ее в буфер обмена и перезагрузите страницу.
        # en: Page has been changed by {author} while you were editing it.
        # Copy your changes to clipboard and reload the page.
        'version_conflict_by': _(
            'Page was edited by {author} while you were editing it.'
            + ' Copy your changes to clipboard and reload the page.'
        ),
        # Translators:
        #  ru: Превышен максимальный размер страницы в {limit} Кб
        #  en: Maximum page size {limit} Kb exceeded
        'too_big_wiki_text': _('Maximum page size {limit} Kb exceeded'),
        # Translators:
        #  ru: Некорректный тег: "{tag}"
        #  en: Bad tag given: "{tag}"
        'bad tag': _('Bad tag given: "{tag}"'),
    }

    @property
    def page_exists(self):
        return self.context['request'].page is not None

    def validate_body(self, body):
        if limit := settings.LIMIT__WIKI_TEXT_FOR_PAGE__BYTES:
            body_size = len(body.encode('utf-8'))
            if body_size > limit:
                page = self.context['request'].page
                logger.warning(f'Page {page.supertag}, body is too big "{body_size}"')
                self.fail('too_big_wiki_text', limit=int(limit / (1 << 10)))
        return body

    def validate_title(self, title):
        if not title and not self.page_exists:
            self.fail('title_for_new_pages_required')
        return title

    def validate_section_id(self, section_id):
        if not self.page_exists and section_id:
            self.fail('no_sections_in_new_page')
        return section_id

    def validate_version(self, version):
        if not version:  # можно сохранять вообще не указывая версию
            return version
        if self.page_exists:
            page = self.context['request'].page
            actual_page_version = page.get_page_version()
            if getattr(page, '_skip_validate_version', False):
                return actual_page_version

            if actual_page_version != version:
                self.fail('version_conflict_by', author=who_edited_the_page(actual_page_version))
        return version

    def validate(self, attrs):
        request = self.context['request']
        if not self.page_exists:
            return attrs

        # проверка типа страницы
        if request.page.page_type == Page.TYPES.GRID:
            self.fail('editing_grid_as_page')
        return attrs

    def save(self):
        request = self.context['request']

        try:
            tags.clean_tag(request.tag)
        except errors.ValidationError:
            raise Http404(_('Bad tag given: "{tag}"').format(tag=request.tag))

        # В бете логика подписок такая:
        # Если страница новая - подписать на нее пользователя-автора страницы.
        # Если страница уже существует, то если пользователь был подписан, оставить его
        # подписанным, а если был не подписан - оставить неподписанным.
        if self.page_exists:
            make_me_watcher = logic_subscription.is_subscribed_to_page(request.user, request.page)
            page_type = None
        else:
            make_me_watcher = True
            page_type = self.data.get('page_type')
            try:
                page_type = Page.TYPES.get_key(page_type)
            except KeyError:
                pass

        try:
            render_page = save_page_from_view(
                request.page,
                request,
                request.tag,
                self.data.get('title'),
                self.data['body'],
                section_id=self.data.get('section_id'),
                authors=get_users_by_identity(self.validated_data.get('authors')),
                make_me_watcher=make_me_watcher,
                page_type=page_type,
            )
        except (Page.DoesNotExist, APIError) as exc:
            logger.warning('Cannot read page "%s" from storage: "%s"', request.supertag, str(exc))
            raise Http404('Can\'t read {0} body in storage'.format(request.supertag))
        except ReservedCreationError:
            raise InvalidDataSentError(_('Bad tag given: "{tag}"').format(tag=request.tag))

        showpage = self.data.get('showpage')
        if showpage:
            showpage_supertag = tag_to_supertag(showpage)
            if showpage_supertag != render_page.supertag:
                render_page = pages_dao.get_page_by_supertag(supertag=showpage_supertag)

        return render_page


class PageCloneSerializer(serializers.Serializer):
    destination = TagField(page_should_exist=False)
