import datetime
import logging
from itertools import chain
from typing import Iterable

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki import access as wiki_access
from wiki.api_core.errors.bad_request import AccessRequestAlreadyProcessed
from wiki.api_core.errors.permissions import AlreadyHasAccessError, UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceIsMissing
from wiki.api_frontend.utils.accessors import department_groups_by_ids, groups_by_ids
from wiki.notifications import signals
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.org import org_user
from wiki.pages.access import get_access_status, get_raw_access, interpret_raw_access, is_admin
from wiki.pages.logic.request_access import get_group_responsibles_for_page
from wiki.pages.models import Access, AccessRequest
from wiki.utils import timezone
from wiki.utils.features.get_features import get_wiki_features

if settings.IS_INTRANET:
    from wiki.pages.access import yandex_group
    from wiki.pages.access.groups import yamoney_group

ACTION_DENY = 'deny'
ACTION_ALLOW_APPLICANT = 'allow_applicant'
ACTION_ALLOW_GROUPS = 'allow_groups'


def set_resolution_deny(access_request: AccessRequest, reason: str, user):
    access_request.verdict = False
    access_request.verdict_reason = reason
    access_request.verdict_by = user
    access_request.save()

    meta = {
        'access_request_id': access_request.pk,
        'resolution_type': 'notchanged',
        'receiver_id': [access_request.applicant_id],
    }
    timeout = timezone.now() + datetime.timedelta(minutes=1)
    notification = PageEvent(
        page=access_request.page,
        author=user,
        timeout=timeout,
        event_type=EventTypes.resolve_access,
        notify=True,
        meta=meta,
    )
    notification.save()


def set_resolution_allow_applicant(access_request, user):
    _set_resolution_allow(access_request, ACTION_ALLOW_APPLICANT, [], user)


def set_resolution_allow_groups(
    access_request,
    all_groups: Iterable['Group'],
    user,
):
    _set_resolution_allow(access_request, ACTION_ALLOW_GROUPS, all_groups, user)


def _set_resolution_allow(
    access_request,
    decision: str,
    all_groups: Iterable['Group'],
    user,
):
    access = interpret_raw_access(get_raw_access(access_request.page))
    allowed_groups = set()
    allowed_users = set()

    if access['is_restricted']:
        for group in access['groups']:
            allowed_groups.add(group)
        for user_with_access in access['users']:
            allowed_users.add(user_with_access)

    if access.get('is_common') or access.get('is_common_wiki'):
        # у группы Яндекс должен остаться доступ WIKI-4217
        allowed_groups.add(yandex_group())
        if yamoney_group():
            allowed_groups.add(yamoney_group())

    if decision == ACTION_ALLOW_APPLICANT:
        access_request.verdict = True
        allowed_users.add(access_request.applicant.staff)
    elif decision == ACTION_ALLOW_GROUPS:
        user_groups_ids = access_request.applicant.get_all_group_ids()

        for group in all_groups:
            allowed_groups.add(group)

            if group.id not in user_groups_ids:
                pe = PageEvent(
                    page=access_request.page,
                    author=user,
                    event_type=EventTypes.request_access,
                    notify=True,
                    timeout=timezone.now() + datetime.timedelta(minutes=1),
                )
                meta_dict = {
                    'resolution_type': 'addtogroup',
                    # когда вы вычитаете из хранилища этот JSON, такой группы может не быть уже
                    # потому что группы можно удалять а то что лежит в хранилище никогда не обновляется
                    'group_id': group.id,
                    'group_name': group.name,
                    'emitter_name': access_request.applicant.username,
                    # это поле не писалось в хранилище до 15 апреля 2017
                    'emitter_uid': access_request.applicant.staff.uid,
                    'access_request_id': access_request.pk,
                    'notify_all': False,
                }
                if get_wiki_features().cloud_uid_supported:
                    if access_request.applicant.cloud_uid:
                        meta_dict['emitter_uid'] = access_request.applicant.cloud_uid
                pe.meta = meta_dict
                pe.save()
            access_request.verdict = True
    else:
        raise ValueError(f'Bad value for decision {decision}')

    try:
        wiki_access.set_access(
            access_request.page,
            wiki_access.TYPES.RESTRICTED,
            user,
            staff_models=allowed_users,
            groups=allowed_groups,
            send_notification_signals=False,
        )
    except wiki_access.NoChanges:
        # пытаемся изменить права доступа, а менять уже нечего,
        # потому что у пользователя есть доступ
        logging.info('User "%s" already has access to "%s"', user.username, access_request.page.supertag)

    if access_request.verdict is not None:
        signals.access_granted.send(
            sender=__file__,
            access=Access(page=access_request.page, staff=access_request.applicant.staff),
            author=user,
            access_request=access_request,
        )
    access_request.verdict_by = user
    access_request.save()


class ProcessAccessRequestSerializer(serializers.Serializer):
    """
    Используется для валидации параметров запроса обработки запроса доступа и обработки.
    """

    action = serializers.ChoiceField(
        choices=(
            (ACTION_ALLOW_APPLICANT, ACTION_ALLOW_APPLICANT),
            (ACTION_ALLOW_GROUPS, ACTION_ALLOW_GROUPS),
            (ACTION_DENY, ACTION_DENY),
        )
    )
    verdict_reason = serializers.CharField(trim_whitespace=True, required=False, allow_blank=True)
    if settings.IS_BUSINESS:
        groups = serializers.ListField(child=serializers.CharField(min_length=1), required=False)
        departments = serializers.ListField(child=serializers.CharField(min_length=1), required=False)
    else:
        groups = serializers.ListField(child=serializers.IntegerField(), required=False)

    default_error_messages = {
        # Translators:
        #  ru: Не переданы группы
        #  en: No groups given
        'groups_required': _('No groups given'),
        # Translators:
        #  ru: Для отказа требуется указать причину
        #  en: Verdict reason is required for denial
        'verdict_for_denial': _('Verdict reason is required for denial'),
        'cannot use departments in intranet': 'You may not use "departments" key in intranet',
    }

    def validate(self, attrs):
        if attrs['action'] not in (ACTION_DENY, ACTION_ALLOW_APPLICANT):
            departments = attrs.get('departments')
            if departments and settings.IS_INTRANET:
                self.fail('cannot use departments in intranet')
            groups = attrs.get('groups', None)
            if settings.IS_BUSINESS:
                if not groups and not departments:
                    self.fail('groups_required')
            elif not groups:
                self.fail('groups_required')
        if attrs['action'] == ACTION_DENY and not attrs.get('verdict_reason'):
            self.fail('verdict_for_denial')
        return attrs

    @staticmethod
    def load(user, request_id):
        try:
            access_request = AccessRequest.objects.select_related('verdict_by').get(pk=request_id)
        except AccessRequest.DoesNotExist:
            raise ResourceIsMissing

        may_allow = user in access_request.page.get_authors() or is_admin(user)
        if not may_allow:
            people_ids = get_group_responsibles_for_page(access_request.page)

            # either user is CENTER or an responsible person of a group, allowed to access
            if user.staff.pk not in people_ids:
                # Translators:
                #  ru: Пользователь должен быть или автором страницы или ответственным за группу
                #  en: User should either be the page author or the group responsible
                raise UserHasNoAccess(_('User should either be the page owner or the group responsible'))

        return access_request, may_allow

    def save(self, user, request_id):
        """
        @type user: django.contrib.auth.User
        """
        access_request, may_allow = ProcessAccessRequestSerializer.load(user, request_id)

        if access_request.verdict_by is not None:
            raise AccessRequestAlreadyProcessed()

        applicant_django_user = org_user().get(username=access_request.applicant.username)
        if get_access_status(access_request.page.supertag, applicant_django_user):
            raise AlreadyHasAccessError

        if self.data['action'] != ACTION_DENY:
            if not may_allow:
                raise UserHasNoAccess('User has no permission to allow access')

            _set_resolution_allow(
                access_request,
                self.data['action'],
                all_groups=chain(
                    groups_by_ids(self.data.get('groups', [])),
                    department_groups_by_ids(self.data.get('departments', [])),
                ),
                user=user,
            )
        else:
            set_resolution_deny(access_request, self.data['verdict_reason'], user)
