
from rest_framework import serializers

from wiki.api_core.errors.permissions import AlreadyHasAccessError
from wiki.notifications import signals
from wiki.pages.models import AccessRequest
from wiki.utils import timezone


class RequestAccessSerializer(serializers.Serializer):
    """
    Используется для валидации параметров запроса доступа и сохранения запроса.
    """

    reason = serializers.CharField(required=True, min_length=1, trim_whitespace=True)

    def save(self, user, page):
        """
        @type user: django.contrib.auth.User
        """
        # Проверяем, что еще доступа нет
        if page.has_access(user, privilege='read'):
            raise AlreadyHasAccessError

        qs = AccessRequest.objects.filter(applicant=user, page=page, verdict_by=None)
        if qs:
            access_request = qs[0]
            access_request.created_at = timezone.now()
            access_request.reason = self.data['reason']
        else:
            access_request = AccessRequest(applicant=user, page=page, reason=self.data['reason'])
        access_request.save()

        signals.request_access_created(access_request)
