"""
Сериализаторы ревизий
"""
import logging

from django.contrib.auth import get_user_model
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.serializers import DateTimeTzAwareField
from wiki.api_frontend.serializers.base import EntityArraySerializer
from wiki.api_frontend.serializers.users import UserProxy, UserSerializer
from wiki.grids.filter import get_column_names
from wiki.notifications.dao import revision_by_page_event
from wiki.notifications.models import PageEvent
from wiki.org import org_user
from wiki.pages.models import Revision
from wiki.utils.context_bound_serializer import ContextBoundSerializer
from wiki.utils.limits import limit__user_max_rev_num
from wiki.utils.serializer_contexts import RequestContext

logger = logging.getLogger(__name__)

PAGEEVENT_TYPES = dict((v, k) for k, v in PageEvent.EVENT_TYPES._asdict().items())


class RevisionItemSerializer(serializers.ModelSerializer):
    """
    Сериализатор ревизии в списке ревизий
    """

    author = UserSerializer()
    created_at = DateTimeTzAwareField()

    class Meta:
        model = Revision
        fields = ('id', 'created_at', 'author')


class RevisionsArraySerializer(EntityArraySerializer):
    """
    Сериализатор для списка ревизий
    """

    def get_data_array(self, page):
        revisions = list(page.revision_set.select_related('author').order_by('-created_at')[self.start : self.stop])

        # закинуть всех используемых пользователей в кэш
        UserProxy.populate_cache([revision.author for revision in revisions])
        return RevisionItemSerializer(revisions, many=True).data

    def get_data_count(self, page):
        return page.revision_set.count()


class PageEventItemSerializer(serializers.Serializer):
    """
    Сериализатор события PageEvent в списке событий
    """

    id = serializers.IntegerField()
    author = UserSerializer()
    event_type = serializers.SerializerMethodField()
    revision = serializers.SerializerMethodField()
    created_at = DateTimeTzAwareField()
    additional_fields = serializers.SerializerMethodField()

    def get_event_type(self, instance):
        event_type = PAGEEVENT_TYPES[instance.event_type]
        object_of_changes = instance.meta.get('object_of_changes')
        if object_of_changes == 'structure':
            # псевдотип для редактирования структуры гридов
            event_type = 'edit_structure'
        return event_type

    def get_revision(self, instance):
        revision = revision_by_page_event(instance)
        if revision:
            return RevisionItemSerializer(revision).data
        return None

    def get_additional_fields(self, event):
        additional_fields = {}

        if event.event_type == PageEvent.EVENT_TYPES.subscribe_other_user and 'subscribed_user_id' in event.meta:
            user_model = get_user_model()
            subscribed_user_id = event.meta['subscribed_user_id']
            try:
                subscribed_user = org_user().select_related('staff').get(id=subscribed_user_id)
                additional_fields['subscribed_user_login'] = subscribed_user.staff.login
                additional_fields['subscribed_user_cloud_uid'] = subscribed_user.cloud_uid
            except user_model.DoesNotExist:
                logger.error("Can't get user with id='%s'" % subscribed_user_id)
        elif event.event_type == PageEvent.EVENT_TYPES.change_owner:
            additional_fields['new_owner'] = event.meta.get('owner_name')
            additional_fields['previous_owner'] = event.meta.get('previous_owner')
            additional_fields['with_subpages'] = event.meta.get('with_children', False)

        return additional_fields


class PageEventsArraySerializer(EntityArraySerializer):
    """
    Сериализатор для списка событий
    """

    default_error_messages = {
        # Translators:
        #  ru: При использовании типа "ограниченный" укажите группу или пользователя
        #  en: When using type "restricted", specify any group or user
        'group_or_user_required': _('When using type "restricted", specify any group or user')
    }

    def __init__(self, *args, **kwargs):
        self.event_types = None
        event_types_text = kwargs.pop('event_types', None)
        if event_types_text:
            self.event_types = self.validate_event_types_param(get_column_names(event_types_text))

        super(PageEventsArraySerializer, self).__init__(*args, **kwargs)

    def get_data_array(self, page):
        qs = page.pageevent_set.select_related('author')
        if self.event_types:
            qs = qs.filter(event_type__in=self.event_types)

        events = list(qs.order_by('-created_at')[self.start : self.stop])
        UserProxy.populate_cache([event.author for event in events])

        events_json = PageEventItemSerializer(events, many=True).data

        # Лимит на число показываемых пользователю событий страницы.
        # Пусть лимит равен 10 и у страницы были следующие события:
        # E E E E E A A A A A F C
        # Мы можем показать только последние 10 событий, т.е.
        # E E E E E A A A A A
        # Если мы выберем отображать событие типа F, то
        # пользователю вернется пустой список, т.к. событие
        # не попало в последине 10 событий.
        limit = limit__user_max_rev_num.get()
        limit_exceeded = False
        if limit is not None and limit != -1:
            # Находим время и дату создания события, начиная с которого
            # мы уже не показываем пользователю события.
            first_hidden_event_creation_dt = page.pageevent_set.order_by('-created_at')[limit : limit + 1].values_list(
                'created_at', flat=True
            )
            if len(first_hidden_event_creation_dt):
                first_hidden_event_dt = first_hidden_event_creation_dt[0]
                cutoff_index = None
                for i, event in enumerate(events):
                    if event.created_at <= first_hidden_event_dt:
                        cutoff_index = i
                        break
                if cutoff_index is not None:
                    events_json = events_json[:cutoff_index]
                    limit_exceeded = True

        return events_json, limit, limit_exceeded

    def get_data_count(self, page):
        qs = page.pageevent_set
        if self.event_types:
            qs = qs.filter(event_type__in=self.event_types)
        return qs.count()

    def to_native(self, obj):
        events, limit, limit_exceeded = self.get_data_array(obj)

        return {
            'data': events,
            'total': self.get_data_count(obj),
            'limit_exceeded': limit_exceeded,
            'limit': limit,
        }

    @staticmethod
    def validate_event_types_param(event_types):
        """
        Проверить значения из списка типов событий: они должны быть из множества допустимых значений,
        определенных в словаре PAGEEVENT_TYPES.
        Если входящий список состоит из наименований типов, соответствующих значениям в словаре PAGEEVENT_TYPES,
        то заменить значения ключами из этого словаря.

        @raise InvalidDataSentError
        """
        try:
            if event_types and ''.join(event_types).replace('_', '').isalpha():
                # если список типов событий содержит имена типов, то заменить их соответствующими
                # цифровыми идентификаторами
                event_types = [
                    list(PAGEEVENT_TYPES.keys())[list(PAGEEVENT_TYPES.values()).index(v)] for v in event_types
                ]
            assert set([int(e) for e in event_types]).issubset(set(PAGEEVENT_TYPES.keys()))
        except (ValueError, AssertionError, KeyError) as exc:
            logger.warning('Wrong event_types parameter value: %s, %s', event_types, repr(exc))
            # Translators:
            #  ru: Неверный параметр event_types: "{0}"
            #  en: Wrong event_types parameter value: "{0}"
            raise InvalidDataSentError(_('Wrong event_types parameter value: "{0}"').format(repr(exc)))

        return event_types


class RevisionSerializer(ContextBoundSerializer[RequestContext], serializers.Serializer):
    id = serializers.ReadOnlyField()
    body = serializers.CharField()
    title = serializers.CharField()
    author = UserSerializer()
    created_at = DateTimeTzAwareField()
