from pydantic import BaseModel
from rest_framework import serializers
from rest_framework.exceptions import ValidationError


class UserIdentity(BaseModel):
    uid: str = None
    cloud_uid: str = None

    @classmethod
    def from_user(cls, user):
        return UserIdentity(uid=user.get_uid(), cloud_uid=user.get_cloud_uid())

    def __repr__(self):
        ids = [self.cloud_uid, self.uid]
        prefixes = ['Cloud-UID: ', 'UID: ']
        out = '|'.join([p + i for i, p in zip(ids, prefixes) if i is not None])
        return f'<{out}>'

    def __eq__(self, other):
        return isinstance(other, UserIdentity) and self.uid == other.uid and self.cloud_uid == other.cloud_uid

    def __hash__(self):
        return self.uid.__hash__() & self.cloud_uid.__hash__()


class UserIdentityField(serializers.Field):
    def to_internal_value(self, data):
        if isinstance(data, dict):
            try:
                val = UserIdentity(**data)
                if val.uid is None and val.cloud_uid is None:
                    raise ValidationError('Identity can not be blank')

                if val.uid is not None and not isinstance(val.uid, str):
                    raise ValidationError('UID must be str')

                if val.cloud_uid is not None and not isinstance(val.cloud_uid, str):
                    raise ValidationError('UID must be str')
                return val
            except TypeError as e:
                raise ValidationError(str(e))

        if isinstance(data, str):
            if len(data) == 0:
                raise ValidationError('UID can not be blank in legacy mode')
            return UserIdentity(uid=data)

        if isinstance(data, int):
            if data < 0:
                raise ValidationError('UID can not be negative')
            return UserIdentity(uid=str(data))

        raise ValidationError('Unsupported format of identity')

    def to_representation(self, value):
        return value.dict()


class UserIdentityListField(serializers.ListField):
    def __init__(self, **kwargs):
        kwargs['child'] = UserIdentityField()
        super().__init__(**kwargs)
