"""
Сериализаторы, относящиеся к пользователям
"""
from django.conf import settings
from rest_framework import serializers

from wiki.api_core.serializers import UserProxy, UserSerializer
from wiki.api_frontend.serializers.base import EntityArraySerializer

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)


class UsersArraySerialiser(EntityArraySerializer):
    def get_attribute(self, instance):
        """
        Возвращаем не атрибут, а весь объект группы.
        """
        return instance

    def get_data_array(self, group):
        """
        Возвращает список пользователей, состоящих в группе
        """
        members_list = group.get_all_members()
        users = members_list[self.start : self.stop]

        # закинуть всех используемых пользователей в кэш
        UserProxy.populate_cache(users)

        return UserSerializer(users, many=True).data

    def get_data_count(self, group):
        return group.get_all_members_count()


class BaseGroupSerializer(serializers.Serializer):
    """
    Базовый сериализатор для групп пользователей
    """

    id = serializers.ReadOnlyField()
    if settings.IS_INTRANET:
        name = serializers.ReadOnlyField()
        url = serializers.SerializerMethodField('get_formatted_url')
        type = serializers.ReadOnlyField()
        externals_count = serializers.ReadOnlyField()
    elif settings.IS_BUSINESS:
        name = serializers.ReadOnlyField(source='title')
        title = serializers.ReadOnlyField()
        dir_id = serializers.ReadOnlyField()
        type = serializers.SerializerMethodField('get_group_type')
    else:
        name = serializers.ReadOnlyField()

    def get_group_type(self, group):
        from wiki.users_biz.models import GROUP_TYPES

        return GROUP_TYPES[group.group_type]

    def get_formatted_url(self, group):
        """
        Сгенерировать урл для группы.
        Для департаментных групп — на Стафф.
        Для сервисных групп — на Планер.
        Для викигрупп — None, потому что мы не поощряем их использование.
        """
        DEPARTMENT_URL_TPL = settings.STAFF_URL + 'departments/{url}/'
        SERVICE_URL_TPL = settings.PLANNER_URL + 'services/{service_id}/'
        if group.is_department:
            return DEPARTMENT_URL_TPL.format(url=group.url)
        elif group.is_service:
            return SERVICE_URL_TPL.format(service_id=group.service_id)


class SimpleGroupSerializer(BaseGroupSerializer):
    """
    Сериализатор группы пользователей с количеством пользователей, состоящих в группе
    """

    users_total = serializers.SerializerMethodField('get_users_total_count')

    def get_users_total_count(self, group):
        return group.get_all_members_count()


class DetailedGroupSerializer(BaseGroupSerializer):
    """
    Сериализатор группы пользователей со списком этих самых пользователей
    """

    users = UsersArraySerialiser()
