"""
Сериализаторы для ответов на ручки подписки
"""
import logging

from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.api_frontend.serializers.user_identity import UserIdentityListField
from wiki.api_v2.exceptions import AlreadyExists
from wiki.pages.logic import subscription
from wiki.subscriptions.exceptions import SubscribeWithoutAccess
from wiki.subscriptions.logic import create_subscription
from wiki.subscriptions.models import SubscriptionType
from wiki.users.dao import get_users_by_identity
from wiki.users.logic.settings import uses_new_subscriptions
from wiki.utils.subscription import generate_event

logger = logging.getLogger(__name__)


class WatchResultSerializer(serializers.Serializer):
    pages_count = serializers.IntegerField(required=False)


class MassWatchInputDataSerializer(WatchResultSerializer):
    # максимально допустимое количество пользователей в одном запросе на подписку.
    MAX_USERS_NUMBER = 100

    default_error_messages = {
        # Translators:
        #  ru: Максимальное число пользователей должно быть не больше {0}
        #  en: The maximum number of users should be no more than {0}
        'max_uids': _('The maximum number of users should be no more than {0}')
    }

    uids = UserIdentityListField(required=False)
    comment = serializers.CharField(required=False, allow_blank=True)

    def validate_uids(self, uids):
        if uids and len(uids) > self.MAX_USERS_NUMBER:
            self.fail('max_uids')

        return uids

    def save(self, author, cluster_page):
        # Количество подписанных страниц
        pages_count = 0
        identities = self.validated_data.get('uids')
        if identities:
            users = get_users_by_identity(identities, panic_if_missing=False)
        else:
            users = [author]

        for user in users:
            if uses_new_subscriptions(user):
                try:
                    create_subscription(
                        user=user,
                        page=cluster_page,
                        type_=SubscriptionType.MY if user.id == author.id else SubscriptionType.OTHER,
                        is_cluster=True,
                    )
                except (AlreadyExists, SubscribeWithoutAccess):
                    continue

                generate_event(
                    'subscribe_other_user' if user.id != author.id else 'watch',
                    author,
                    cluster_page,
                    cluster=[cluster_page.supertag],
                    other_subscribed_user=user if user.id != author.id else None,
                    comment=self.data.get('comment'),
                )
                pages_count += 1
            else:
                pages_count += subscription.subscribe_user_to_cluster(
                    user,
                    cluster_page,
                    author,
                    self.data.get('comment'),
                )

        return pages_count
