import logging
import time
from itertools import chain

from django.conf import settings
from django.contrib.auth import get_user_model

from wiki import access as wiki_access
from wiki.access import TYPES
from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import API_ACCESS_TYPES
from wiki.api_frontend.serializers.access import (
    AccessData,
    AccessModificationSerializer,
    DetailedAccessSerializer,
    DetailedAccessSerializerForPOST,
)
from wiki.api_frontend.utils.accessors import department_groups_by_ids, groups_by_ids
from wiki.api_v2.public.pages.exceptions import EmptyCustomAcl
from wiki.api_v2.public.pages.schemas import GroupSchema
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.integrations.ms.wiki_to_retriever_sqs import RETRIEVER_SQS_CLIENT
from wiki.org import get_org, org_user
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access, is_admin
from wiki.pages.access.consts import LegacyAcl
from wiki.pages.access.exceptions import OutstaffRuleViolation, GroupsRemovedFromIdm
from wiki.pages.dao import subscription
from wiki.pages.models import Page, PageWatch
from wiki.pages.models.consts import AclType
from wiki.users import i18n as users_i18n
from wiki.users.dao import get_users_by_identity
from wiki.users.user_data_repository import USER_DATA_REPOSITORY

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

logger = logging.getLogger(__name__)

User = get_user_model()


def check_page_watches(page_watches):
    _users = org_user().filter(username__in=[_.user for _ in page_watches])
    watchers = dict([(u.username, u) for u in _users])
    page_watches_to_delete = []
    for page_watch in page_watches:
        if watchers.get(page_watch.user, None) is None:
            page_watches_to_delete.append(page_watch.id)
        elif not page_watch.page.has_access(watchers[page_watch.user]):
            page_watches_to_delete.append(page_watch.id)
    PageWatch.objects.filter(id__in=page_watches_to_delete).delete()


class AccessView(PageAPIView):
    """
    View для работы с доступами к странице.
    Тут смешивается бизнес-логика с вью-логикой,
    пока переписал с разделением, но старое не трогал
    """

    serializer_class = AccessModificationSerializer
    render_blank_form_for_methods = ('PUT',)
    available_for_admins = True
    check_readonly_mode = True

    def check_page_access(self):
        """
        Проверить возможность доступа пользователя из запроса к управлению правами страницы
        """
        # управлять доступом могут лишь авторы страницы или админы, но смотреть могут все
        if self.request.method.lower() == 'get':
            super(AccessView, self).check_page_access()
        elif not (self.request.user in self.request.page.get_authors() or is_admin(self.request.user)):
            raise UserHasNoAccess()

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить права доступа к странице.

        %%
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.access"
        %%

        %%(js)
        {
          "data": {
            "type": "inherited",
            "parent_access": "common",
            "opened_to_external_flag": false,  // поле есть только в интранете
            "authors": [{ ... },],
            "restrictions": null
          }
        }
        %%

        type:
            inherited — наследован от родителя
            common — всем сотрудникам
            restricted — ограниченный для пользователей/групп
            owner — только для автора страницы
            anonymous — анонимный

        parent_access
            null — у страницы нет родителя
            common — всем сотрудникам
            restricted — ограниченный для пользователей/групп
            owner — только для автора страницы
            anonymous — анонимный

        restrictions — в случае type="restricted" содержит два списка
            users и groups, в которых перечислены пользователи и группы, у
            которых есть доступ к странице

        Для страниц верхнего уровня не может быть типа доступа inherited

        """
        access_data = AccessData(request.page)
        return self.build_response(
            serializer_class=DetailedAccessSerializer,
            instance=access_data,
        )

    @raises(InvalidDataSentError)
    def post(self, request, *args, **kwargs):
        """
        Изменить права доступа к странице.

        Примеры запросов:

        Установить доступ к странице "только авторы страницы".

        %%
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.access" \
         --data '{"type": "owner"}'
        %%

        %%(js)
        {
            "type": "owner"
        }
        %%

        Установить доступ к странице "ограничен"
          * пользователю c uid "1120000000011111" и "112000000002222"
          * группам с id 4 и 5
          * департаментам с id 6 и 7 (используется только для Вики B2B)

        %%
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.access" \
         --data '{"type": "restricted", "users": ["1120000000011111", "112000000002222"], "groups": [4, 5],
          "departments": [6, 7]}'
        %%

        %%(js)
        {
          "data": {
            "type": "restricted",
            "parent_access": "common",
            "opened_to_external_flag": false,  // поле есть только в интранете
            "authors": [{ ... },],
            "restrictions": {
              "users": [...],
              "groups": [...],
            },
            "rejects": {
              "users": [...],
              "groups": [...],
            },
            "request_role_url": <string> | null
          }
        }
        %%
        """
        validated_data = self.validate()

        logger.info(
            'Page access modification to "%s" for %s/%s/%s'
            % (
                validated_data['type'],
                validated_data.get('users'),
                validated_data.get('groups'),
                validated_data.get('departments'),
            )
        )

        users = get_users_by_identity(validated_data.get('users'), prefetch_staff=True, panic_if_missing=True)

        staff_models = [user.staff for user in users]

        groups = groups_by_ids(validated_data.get('groups'))
        departments = department_groups_by_ids(validated_data.get('departments'))

        if settings.IS_INTRANET and departments:
            # В Интранете Staff и ABC группы передаются в поле "groups",
            # на это рассчитывает код split_by_possibility_of_access_grant()
            raise InvalidDataSentError('"departments" in intranet must be empty')

        all_groups = list(chain(groups, departments))

        ts_0 = time.time()

        access_data = AccessData(request.page)

        # вычислим пользователей и группы в списке acl, которые были добавлены или удалены
        current_user_map, current_groups_map = {}, {}
        if access_data.restrictions:
            current_user_map = {user.id: user for user in access_data.restrictions['users']}
            current_groups_map = {group.id: group for group in access_data.restrictions['groups']}

        request_users_map = {user.id: user for user in users}
        request_groups_map = {group.id: group for group in all_groups}

        added_user_id = request_users_map.keys() - current_user_map.keys()
        added_groups_id = request_groups_map.keys() - current_groups_map.keys()

        added_staff_models = [request_users_map.get(idx).staff for idx in added_user_id]
        added_groups_models = [request_groups_map.get(group_id) for group_id in added_groups_id]

        outstaff_staff_models = wiki_access.extract_outstaff(added_staff_models)
        outstaff_groups, non_existent_groups = wiki_access.extract_outstaff_and_deleted_groups(added_groups_models)

        is_outstaff_manager = wiki_access.is_outstaff_manager(request.user)

        # Если автор запроса не имеет роли outstaffmanager, то сохранить изменения в настройках доступа он
        # может только в том случае, если он не добавлял и не удалял Outstaff пользователей или группы
        # пользователей без роли external.
        if (len(outstaff_groups) > 0 or len(outstaff_staff_models) > 0) and not is_outstaff_manager:
            outstaff_users = [staff.user for staff in outstaff_staff_models]

            logging.info(
                'Can\'t save access rights to page because some users or groups were rejected: %s, %s'
                % (outstaff_users, outstaff_groups)
            )
            access_data.rejects = {'users': outstaff_users, 'groups': outstaff_groups}
            access_data.request_role_url = wiki_access.get_outstaff_manager_idm_role_request(request.user)

            return self.build_response(
                serializer_class=DetailedAccessSerializerForPOST,
                instance=access_data,
            )

        # Если попытаться добавить удаленную группу, это должно привести к ошибке
        # При этом мы разрешаем держать удаленные группы в доступах просто их не показываем
        # non_existent_groups = added + removed
        added_non_existent_groups = [grp for grp in non_existent_groups if grp.id in added_groups_id]

        if len(added_non_existent_groups) > 0:
            raise InvalidDataSentError(users_i18n.groups_with_names_have_deleted_status(added_non_existent_groups))

        ts_1 = time.time()

        try:
            wiki_access.set_access(
                page=request.page,
                access_type=API_ACCESS_TYPES[validated_data['type']],
                author_of_changes=request.user,
                staff_models=staff_models,
                groups=all_groups,
            )

            # Часть сервиса cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_update_acl(request.page)
            # Отправка сообщения в retriever
            RETRIEVER_SQS_CLIENT.on_model_update_acl(request.page)

        except wiki_access.NoChanges:
            # никаких изменений в правах доступа. Это допустимо.
            logging.info('User made no changes to access: page="%s", user="%s"', request.page.id, request.user.username)

        except wiki_access.NoUsersAndGroups:
            # Происходит если мы отклонили все группы и всех пользователей, которых запрашивал пользователь
            # Пример - хочет дать доступ аутстафферам, но сам не может давать доступ аутстафферам
            # Наверное тут надо выводить какое-то предупреждение именно пользователю
            # Но это вполне рабочий кейс
            logging.info('Looks like we rejected every grantee;')

        ts_2 = time.time()
        # обновим список смотрящих за страницей
        self._refresh_page_watchers()

        access_data = AccessData(request.page)
        access_data.rejects = {'users': [], 'groups': []}
        access_data.request_role_url = None

        ts_3 = time.time()
        if settings.IS_INTRANET:
            self._refresh_access_status(access_data)

        ts_4 = time.time()

        logger.info(
            'Access modification duration breakdown: %.2f ms / %.2f ms / %.2f ms / %.2f ms'
            % (
                (ts_1 - ts_0) * 1000,
                (ts_2 - ts_1) * 1000,
                (ts_3 - ts_2) * 1000,
                (ts_4 - ts_3) * 1000,
            )
        )

        return self.build_response(
            serializer_class=DetailedAccessSerializerForPOST,
            instance=access_data,
        )

    def _refresh_page_watchers(self):
        """
        Обновить список подписчиков страницы в соответствие с изменившимися правами доступа
        """
        page = self.request.page
        access_changed = {page}
        pages_to_check = list(
            Page.objects.filter(supertag__startswith=page.supertag + '/', org=get_org()).only('supertag')
        )
        access_list = get_bulk_raw_access(pages_to_check)
        for page_to_check, raw_access in access_list.items():
            interpretation = interpret_raw_access(raw_access)
            if interpretation['is_inherited']:
                if raw_access['latest_supertag'].startswith(page.supertag + '/'):
                    continue
                access_changed.add(page_to_check)
        page_watches = chain(*list(subscription.filter_pages_watches(access_changed).values()))
        check_page_watches(page_watches)

    def _refresh_access_status(self, access_data):
        page = self.request.page
        current_external_access = page.opened_to_external_flag
        new_access_status = access_data.opened_to_external_flag

        if current_external_access != new_access_status:
            page = Page.objects.get(id=page.id)
            page.opened_to_external_flag = new_access_status
            page.save()


"""
Вынесенная бизнес-логика живет тут:
"""

ACL_TYPE_TO_ACCESS_TYPE = {
    AclType.CUSTOM: TYPES.RESTRICTED,
    AclType.DEFAULT: TYPES.COMMON,
    AclType.ONLY_AUTHORS: TYPES.OWNER,
}


def validate_acl_request(user, page_acl: LegacyAcl, desired_acl: LegacyAcl):
    """
    Если автор запроса не имеет роли outstaffmanager, то сохранить изменения в настройках доступа он
    может только в том случае, если он не добавлял и не удалял Outstaff пользователей или группы
    пользователей без роли external.

    Некоторые группы могут пропасть из IDM, и мы не должны разрешать их выставление
    """

    added_users = set(desired_acl.users) - set(page_acl.users)
    added_groups = set(desired_acl.groups) - set(page_acl.groups)

    outstaff_staff_models = wiki_access.extract_outstaff([user.staff for user in added_users])
    outstaff_groups, non_existent_groups = wiki_access.extract_outstaff_and_deleted_groups(added_groups)

    if len(non_existent_groups) > 0:
        raise GroupsRemovedFromIdm(
            details={'missing_groups': [GroupSchema.serialize(group) for group in non_existent_groups]}
        )

    is_outstaff_manager = wiki_access.is_outstaff_manager(user)

    if (len(outstaff_groups) > 0 or len(outstaff_staff_models) > 0) and not is_outstaff_manager:
        raise OutstaffRuleViolation(
            details={
                'outstaff_users': [
                    USER_DATA_REPOSITORY.orm_to_user_schema(staff.user) for staff in outstaff_staff_models
                ],
                'outstaff_groups': [GroupSchema.serialize(group) for group in outstaff_groups],
                'idm_role_url': wiki_access.get_outstaff_manager_idm_role_request(user),
            }
        )


def apply_legacy_acl(page, user, desired_acl: LegacyAcl):

    if not desired_acl.break_inheritance:
        access_type = TYPES.INHERITED
    else:
        access_type = ACL_TYPE_TO_ACCESS_TYPE[desired_acl.acl_type]

    try:
        wiki_access.set_access(
            page=page,
            access_type=access_type,
            author_of_changes=user,
            staff_models=[user.staff for user in desired_acl.users],
            groups=desired_acl.groups,
        )
    except wiki_access.NoChanges:
        pass
    except wiki_access.NoUsersAndGroups:
        raise EmptyCustomAcl()


def apply_legacy_acl_sideeffects(page):
    _side_effect_refresh_access_status(page)
    _side_effect_notify_queues(page)
    _refresh_page_watchers(page)


def _side_effect_notify_queues(page):
    # Часть сервиса cloudsearch
    CLOUD_SEARCH_CLIENT.on_model_update_acl(page)
    # Отправка сообщения в retriever
    RETRIEVER_SQS_CLIENT.on_model_update_acl(page)


def _side_effect_refresh_access_status(page):
    if not settings.IS_INTRANET:
        return

    access_data = AccessData(page)
    access_data.rejects = {'users': [], 'groups': []}
    access_data.request_role_url = None

    current_external_access = page.opened_to_external_flag
    new_access_status = access_data.opened_to_external_flag

    if current_external_access != new_access_status:
        page = Page.objects.get(id=page.id)
        page.opened_to_external_flag = new_access_status
        page.save()


def _refresh_page_watchers(page):
    """
    Обновить список подписчиков страницы в соответствие с изменившимися правами доступа
    """
    access_changed = {page}
    pages_to_check = list(Page.objects.filter(supertag__startswith=page.supertag + '/', org=get_org()).only('supertag'))
    access_list = get_bulk_raw_access(pages_to_check)
    for page_to_check, raw_access in access_list.items():
        interpretation = interpret_raw_access(raw_access)
        if interpretation['is_inherited']:
            if raw_access['latest_supertag'].startswith(page.supertag + '/'):
                continue
            access_changed.add(page_to_check)
    page_watches = chain(*list(subscription.filter_pages_watches(access_changed).values()))
    check_page_watches(page_watches)
