from django.db import transaction
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.actuality import ActualityMarkDeserializer, ActualityMarkSerializer
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.pages.logic.actuality import get_page_actuality_details, mark_page_actuality
from wiki.utils.errors import ValidationError as LogicValidationError


class ActualityView(PageAPIView):
    """
    Получение деталей и изменение отметки об актуальности страницы.
    """

    serializer_class = ActualityMarkDeserializer
    check_readonly_mode = True

    @raises()
    @transaction.atomic
    def get(self, request, *args, **kwargs):
        """
        Получить детали отметки об актуальности/устаревшести страницы.

        %%
        curl -H "Authorization: OAuth <token>" -XGET "https://wiki-api.yandex-team.ru/_api/frontend/page/.actuality"
        %%
        Тело запроса пустое.

        Тело ответа:
        %%(js)
        {
            "status": "obsolete", // "obsolete" или "actual"
            "marked_at": "2014-06-04T20:50:29",  // дата помечания актуальной/устаревшей
            "user": {
                // данные про пометившего пользователя в стандартном формате
            },
            "comment": ,  // комментарий к пометке, такая же структура, как в ответе редактирования страницы,
                          // может быть null
            "links": [  // список относительных ссылок на более актуальные страницы,
                        //или абсолютных url на внешние ресурсы с актуальной информацией, может быть пустым
                "/some/page",
                "/another/page",
                "https://yandex.ru"
            ]
        }
        %%
        Если у страницы нет отметки об актуальности/устаревшести, то ответ будет пустым.
        Если отметка есть, то при конкурентных изменениях значение статуса может оказаться отличным от значения при
        получении данных страницы.
        """

        page = request.page
        actuality_details = get_page_actuality_details(page, request)
        if actuality_details is None:
            return Response(EmptySerializer().data)

        return Response(
            ActualityMarkSerializer(
                {
                    'status': page.actuality_status,
                    'marked_at': page.actuality_marked_at,
                    'user': actuality_details.mark.user,
                    'comment': actuality_details.comment_bemjson,
                    'links': actuality_details.links,
                }
            ).data
        )

    @raises()
    @transaction.atomic
    def post(self, request, *args, **kwargs):
        """
        Пометить страницу актуальной/устаревшей.

        Пример запроса для страницы /page:
        %%
        curl -H "Authorization: OAuth <token>" -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/page/.actuality"
        %%
        Тело запроса:
        %%(js)
        {
            "actual": True, // True для пометки актуальной, False для пометки устаревшей
            "comment": "sometext",  // комментарий к пометке, может быть пустой строкой или null
            "links": [  // список ссылок на более актуальные страницы, может быть пустым,
                        // должен быть пустым при actual=True
                "/some/page",
                "wiki.yandex-team.ru/another/page"
                "http://wiki.yandex-team.ru/page1"
                "//old.wiki.yandex-team.ru/page2"
            ]
        }
        %%

        Если нет ошибок, тело ответа будет пустым.
        Каждый УРЛ в списке актуальных страниц будет валидироваться.
        """
        data = self.validate()
        comment = data.get('comment')
        if comment == '':
            comment = None
        try:
            mark_page_actuality(
                user=request.user,
                page=request.page,
                is_actual=data['actual'],
                comment=comment,
                mixed_links=data.get('links') or [],
            )
        except LogicValidationError as e:
            raise InvalidDataSentError(
                # Translators:
                #  ru: Измените пожалуйста эту ссылку "%s"
                #  en: Please change this link "%s"
                _('Please change this link "%s"')
                % e.invalid_value,
            )

        return Response(EmptySerializer().data)
