from django_pgaas import atomic_retry

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView, raises
from wiki.api_frontend.serializers.authors import AuthorsSerializer, ChangeAuthorsSerializer
from wiki.pages.logic import authors as authors_logic
from wiki.users.dao import get_users_by_identity


class AuthorsView(PageAPIView):
    """
    Управление авторами страницы.
    """

    serializer_class = ChangeAuthorsSerializer
    check_readonly_mode = True

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить список авторов страницы.

        %%
        curl -H "Authorization: OAuth <token>"
        "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.authors"
        %%

        Ответ
        %%(js)
        {
            "authors": [
                {...},  // пользователь в стандартном формате
            ]
        }
        %%
        """

        query_authors = request.page.authors.order_by('staff__is_dismissed', 'id').select_related('staff')

        if owner := request.page.owner:
            co_authors = query_authors.exclude(id=owner.id)
            authors = [owner] + list(co_authors)
        else:
            authors = query_authors.all()

        return self.build_response(
            serializer_class=AuthorsSerializer,
            instance={'authors': authors},
        )

    @raises(UserHasNoAccess)
    @atomic_retry()
    def post(self, request, *args, **kwargs):
        """
        Изменить авторов страницы

        Изменить список авторов текущей страницы путем замены существующего списка значениями
        из переданного списка "authors".
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.authors"
        %%
        Тело запроса:
        %%(js)
        {
            "authors": [
                "1120000000000528",
                "1120000000000529",
            ]
        }
        %%

        Изменить список авторов текущей страницы путем замены существующего списка значениями
        из переданного списка "authors", а также изменить список авторов в подстраницах, где в соавторах есть все авторы
        текущей страницы, путем замены авторов из списка текущей страницы на авторов из переданного списка "authors"
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.authors"
        %%
        Тело запроса:
        %%(js)
        {
            "authors": [
                "1120000000000528",
                "1120000000000529",
            ]
            "with_subpages": true
        }
        %%

        Изменить список авторов в кластере, включая текущую страницу, путем замены авторов из переданного
        списка "replace_authors" на авторов из переданного списка "authors".
        %%
        curl -H "Authorization: OAuth <token>"
        -XPOST "https://wiki-api.yandex-team.ru/_api/frontend/somepage/.authors"
        %%
        Тело запроса:
        %%(js)
        {
            "authors": [
                "1120000000000528",
                "1120000000000529",
            ],
            "replace_authors": [
                "1120000000000529",
                "1120000000009258",
            ],
            "with_subpages": true
        }
        %%
        """
        page = request.page
        user = request.user

        data = self.validate()
        authors = get_users_by_identity(data.get('authors'), panic_if_missing=True)
        replace_authors = get_users_by_identity(data.get('replace_authors'), panic_if_missing=True)
        with_subpages = data['with_subpages']

        if with_subpages:
            authors_logic.change_authors_for_cluster(
                root=page,
                user=user,
                authors=authors,
                replace_authors=replace_authors,
            )
        else:
            if not authors_logic.can_change_authors(page, user):
                raise UserHasNoAccess('Only authors of page can change list of page authors')

            authors_logic.change_authors_for_page(page=page, user=user, authors=authors)

        return self.build_success_status_response()

    def check_page_access(self):
        pass
