"""
View для различных автокомплитов
"""
import urllib.parse

from django.conf import settings
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import Response, WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import paginated
from wiki.api_frontend.serializers.autocomplete import (
    AuthAutocompleteDeserializer,
    AuthAutocompleteSerializer,
    AutocompletePageArraySerializer,
)
from wiki.users.dao import find_staff_models, find_wiki_group_models


class PageSupertagAutocompleteView(WikiAPIView):
    """
    View для показа списка подстраниц
    """

    @raises()
    @paginated
    def get(self, request, start=None, limit=None, *args, **kwargs):
        """
        Показывает список подстраниц по переданному GET-параметру "tag".

        %%(bash)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.autocomplete?tag=jandeks"
        %%

        %%(json)
        {
            "data": {
                "total": 2,
                "data": [
                    {"user_access": "allowed-common", "supertag": "sales/ac", "tag": "Салес/АЦ", "url": "/Sales/AC"},
                    {"user_access": "allowed-common", "supertag": "sales/ad", "tag": "Салес/АД", "url": "/Sales/AD"}
                     ...
            ]}
        }
        %%
        """
        tag = self.request.GET.get('tag')
        if not tag or len(tag) < 3:
            raise InvalidDataSentError(
                # Translators:
                #  ru: Адрес страницы должен быть 3 символа как минимум
                #  en: Page address must be 3 symbols or longer
                _('Page address must be 3 symbols or longer')
            )

        # обрабатываем ситуацию, когда в значении параметра tag передается абсолютный url
        tag = urllib.parse.urlparse(tag).path.lstrip('/')

        # TODO: возвращать page-like object
        return Response(
            AutocompletePageArraySerializer(tag, context={'user': self.request.user}, start=start, limit=limit).data
        )


class AuthAutocompleteView(WikiAPIView):
    """
    View для поиска пользователя и/или группы.
    """

    TOTAL_RESULT_LIMIT = 8

    serializer_class = AuthAutocompleteDeserializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Возвращает список пользователей и/или групп по переданной строке.
        Строка передаётся в обязательном GET-параметре "q", не может быть пустой.
        В опциональном GET-параметре "filter" можно передать строку "user" или "group", чтобы возвращались
        только пользователи или только группы, иначе будут возвращаться и группы, и пользователи.

        %%(bash)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.autocomplete/auth?q=a"
        %%

        %%(json)
        {
            "data": {
                "users": [
                    {"id": 23, "uid": "100500", "login": "vasily", "first_name": "Vasily", "last_name": "Pupkin"}
                    ...
                ],
                "groups": [
                    {"id": 8, "name": "Admins"},
                     ...
                ]
            }
        }
        %%
        """
        assert not settings.IS_INTRANET, 'this functionality is not for Intranet'

        deserializer = self.serializer_class(data=request.GET)
        if not deserializer.is_valid():
            raise InvalidDataSentError(deserializer.errors)

        data = deserializer.validated_data

        users = groups = None
        if data.get('filter') != 'group':
            users = find_staff_models(data.get('q'), limit=self.TOTAL_RESULT_LIMIT)
        if data.get('filter') != 'user':
            groups = find_wiki_group_models(data.get('q'), limit=self.TOTAL_RESULT_LIMIT)

        return Response(AuthAutocompleteSerializer({'users': users, 'groups': groups}).data)
