

import logging

from django.conf import settings
from rest_framework.response import Response

from wiki.actions import get_action_class
from wiki.actions.classes.base_action import ActionError
from wiki.api_core.framework import PageAPIView, raises
from wiki.api_core.utils import is_tvm_authentication
from wiki.utils.wiki_robot import get_wiki_robot

logger = logging.getLogger(__name__)


class WikiActionView(PageAPIView):
    def check_page_access(self):
        if (
            is_tvm_authentication(self.request)
            and str(self.request.yauser.service_ticket.src) in settings.PAGE_SEARCH_INDEX_ACCESS_TVM_CLIENT_IDS
        ):
            # Если пришел Поиск за данными, то доступы к страницам не проверяем и все динамические экшены выполняем
            # от имени нашего робота
            self.request.user = get_wiki_robot()
        else:
            super(WikiActionView, self).check_page_access()

    @raises(ActionError)
    def get(self, request, *args, **kwargs):
        """
        Возвращает результат работы экшена в виде JSON.

        Параметры, передаваемые экшену имеют вид

        %%(json)
        {
            "action_name": "action_name",
            "url_with_data": /..../
        }
        %%
        где параметры начинающиеся с "_" - это параметры передаваемые в экшен,
        а все остальные параметры, сколько бы их ни было, имеют служебное назначение.
        """
        action_name = request.GET['action_name']

        try:
            action_class = get_action_class(action_name)
        except ImportError:
            logger.exception('Cannot import action "%s" for page "%s"', action_name, request.page.supertag)
            raise
        action = action_class(params=None, request=request)
        params_from_frontend = request.GET
        return Response(action.render_json_for_get(params=params_from_frontend))
