import logging

from django.db import OperationalError
from django.http import Http404
from django.utils.translation import ugettext_lazy as _
from django_pgaas import atomic_retry
from psycopg2 import errorcodes as pg_errorcodes
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.hacks.grid_hacks import replace_width_in_grid_structure
from wiki.api_core.raises import raises
from wiki.api_frontend.logic import GridProxy
from wiki.api_frontend.logic import errors as errors_logic
from wiki.api_frontend.serializers.grids import FullGridVersionedStatusSerializer, ModifyingSequenceSerializer, errors
from wiki.api_frontend.serializers.grids.io import CellVersionedStatusSerializer
from wiki.api_frontend.serializers.io import VersionedStatusSerializer
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.grids.models import Grid
from wiki.users.core import get_support_contact

logger = logging.getLogger(__name__)


class DefaultAnswerProxy(object):
    _grid = None
    _success = None
    _message = None

    def __init__(self, user_auth, grid, success, message=None):
        self._grid = grid
        self._user_auth = user_auth
        self._success = success
        self._message = message

    @property
    def success(self):
        return self._success

    @property
    def message(self):
        return self._message

    def __getattr__(self, item):
        return getattr(self._grid, item)


class ChangeGridView(PageAPIView):
    """
    Позволяет изменять данные и структуру грида.
    """

    _grid = None

    serializer_class = ModifyingSequenceSerializer
    successful_answer_proxy = DefaultAnswerProxy
    successful_response_serializer = VersionedStatusSerializer
    check_readonly_mode = True

    @atomic_retry()
    @raises(
        errors.NoSuchRowError,
        errors.NoSuchColumnError,
        errors.CellHasChangedError,
        errors.ColumnHasChangedError,
        errors.TitleHasChangedError,
        errors.ConflictingChangeError,
        errors.SortingHasChangedError,
        errors.NoSuchOptionError,
        errors.DuplicateOptionError,
        errors.NoMoveTargetRowsError,
        errors.TicketDependentFieldWithoutTicket,
        errors.SortingHasChangedError,
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
        errors.ObjectIsLockedForUpdate,
    )
    def post(self, request, *args, **kwargs):
        """
        Изменить табличный список.
        {{a name="grid-change-top"}}

        Здесь видно, что changes - это список, поэтому вы можете передавать
        сразу несколько следующих друг за другом изменений, они будут наложены
        последовательно.

          * ((#grid-add-row добавление строки))
          * ((#grid-move-row перемещение строки))
          * ((#grid-remove-row удаление строки))
          * ((#grid-edited-row редактирование строки))
          * ((#grid-create-column создание столбца))
          * ((#grid-change-column изменение столбца))
          * ((#grid-move-column перемещение столбца))
          * ((#grid-remove-column удаление столбца))
          * ((#grid-change-title изменение названия табличного списка))
          * {{#grid-change-select-column изменение/добавление столбца типа Список/Множественный список}}

        {{a name="grid-move-column"}}
        Пример **запроса, перемещающего столбец** в начало таблицы:

        %%(js)
        {
          "version": "10",
          "changes": [{
              "column_moved": {
                  "id": "1",
                  "after_id": "5"
              }
            }
          ]
        }
        %%

        Ответ:


        %%(js)
        {
          "data": {
              "success": true,
              "version": "11",
          }
        }
        %%
        after_id="-1" означает сделать первым столбцом.

        ((#grid-change-top назад к списку))

        {{a name="grid-edited-row"}}
        Пример **запроса, изменяющего существующую строку**:

        %%(js)
        {
          "version": "10",
          "changes": [
            {
              "edited_row": {
                  "id": "1", // индекс строки
                  "data": { // Этот мап передавать обязательно.
                    // Здесь должны быть перечислены имена столбцов и их новые значения.
                    // Передача значений описана в секции про добавление строки.
                    "latest_film": "Boston Legal" // пример
                  }
              }
            }
          ]
        }
        %%

        Ответ:


        %%(js)
        {
          "data": {
              "success": true,
              "version": "123",
          }
        }
        %%

        ((#grid-change-top назад к списку))

        {{a name="grid-move-row"}}
        Пример запроса, перемещающего 2 строки за раз.

        %%(js)
        {
          "version": "100",
          "changes": [
            {
              "row_moved": {
                  "id": "17",
                  "after_id": "10",
                  "before_id": "23"
              }
            },
            {
              "row_moved": {
                  "id": "5",
                  "after_id": "-1"
                  "before_id": "10"
              }
            },
          ]
        }
        %%
        -1 в after_id означает сделать первой строкой, -1 в before_id означает сделать последней строкой.
        ((#grid-change-top назад к списку))

        {{a name="grid-create-column"}}
        Пример **запроса, создающего столбец**:
        %%(js)
        {
          "version": "100",
          "changes": [
            {
              "added_column": {
                "type": "string",  // обязательный параметр
                "title": "Любимый герой сериала",  // обязательный параметр
                "width": 100,  // может быть null
                "width_units": "percent",  # или "pixel", или "percent"
                // required - это пожелание, а не требование.
                // данное поле заполнять необязательно
                "required": false,
                "options": null,  // используется только полями типа select
                "markdone": true,  // используется только полями типа checkbox
              }
            }
          ]
        }
        %%

        Столбец всегда добавляется в конец грида.
        ((#grid-change-top назад к списку))

        {{a name="grid-add-row"}}
        Пример запроса, **добавляющего строку** перед всеми (в начало):
        В табличном списке есть поля latest_film, number, year_of_birth,
        cool_actor. Тогда для сохранения надо послать такой запрос:

        %%(js)
        {
          "version": "43",
          "changes": [{
            "added_row": {
              "after_id": "10",
              "before_id": "-1",
              "data": {
                // перечислить набор полей, которые надо заполнить
                // если ничего заполнять не надо - передайте data: {}
                "latest_film": "Boston Legal",  // тип поля - строка
                "number": 10,                   // тип поля число
                "year_of_birth": "1950",        // тип поля список, также можно передать ["1950"]
                "languages": ["ru", "en"],      // тип поля множественный список
                "cool_actor": true,             // тип поля список
              }
            }
          }]
        }
        %%

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%after_id%% | str | !!обязательный!! | индекс строки, после которой требуется вставить новую.
        Значение "-1" означает вставить строку первой. Несуществующий индекс игнорируется.
        Например, after_id="last" будет означать вставить строку последней  ||
        || %%before_id%% | str | опциональный | индекс строки, перед которой требуется вставить новую.
        Значение "-1" означает вставить строку последней. Значение по умолчанию: "-1" ||
        |#

        -1 в after_id означает сделать первой строкой, -1 в before_id означает сделать последней строкой.
        ((#grid-change-top назад к списку))

        {{a name="grid-change-column"}}
        Пример **запроса, изменяющего столбец**. В запросе надо перечислять все
        поля, которые составляют описание столбца, даже если вы хотите изменить
        только некоторые из них.

        Тип столбца менять нельзя. Все поля аналогичны запросу добавление
        столбца, кроме name - его требуется указывать явно.

        %%(js)
        {
          "version": "43",
          "changes": [{
            "edited_column": {
              // имя берется из структуры грида:
              // grid["structure"]["fields"][int]["name"]
              "name": "имя столбца",
              // required - это пожелание, а не требование.
              // данное поле в строке заполнять необязательно
              "required": true,
              "width": 20,
              "width_units": "pixel",
              "title": "Роль"
            }
          }]
        }
        %%
        ((#grid-change-top назад к списку))

        Если есть строки с пустыми значениями в столбце 1, то ответ будет таким
        %%(js)
        {
          "version": "43",
          "status": false,
          "message": "CannotChangeColumnToRequired"  // Это сообщение может измениться.
        }
        %%
        Если все корректно, то ответ будет таким:

        %%(js)
        {
          "version": "53",
          "status": true
        }
        %%
        ((#grid-change-top назад к списку))

        {{a name="grid-remove-column"}}
        Пример **запроса, удаляющего столбец** в начале грида:

        %%(js)
        {
            "version": "123",
            "changes": [
                {
                    "removed_column": {
                        "name": "имя столбца"
                    }
                }
            ]
        }
        %%

        Можно прислать запрос на удаление несуществующего столбца.
        Это корректно и в ответ будет 200.

        Ответ:

        %%(js)
        {
            "data": {
                "success": true,
                "version": "123",
            }
        }
        %%

        ((#grid-change-top назад к списку))

        {{a name="grid-remove-row"}}
        Пример **запроса, удаляющего строку** в начале грида:

        %%(js)
        {
            "version": "10",
            "changes": [
                {
                    "removed_row": {
                        "id": "0"
                    }
                }
            ]
        }
        %%

        Ответ:

        %%(js)
        {
            "data": {
                "success": true,
                "version": "10",
            }
        }
        %%

        ((#grid-change-top назад к списку))

        {{a name="grid-change-title"}}
        Пример **запроса, меняющего название**:

        %%(js)
        {
            "version": "10",
            "changes": [
                {
                    "title_changed": {
                        "title": "новое название"
                    }
                }
            ]
        }
        %%

        Ответ:

        %%(js)
        {
            "data": {
                "success": true,
                "version": "10",
            }
        }
        %%

        ((#grid-change-top назад к списку))

        {{a name="grid-change-sorting"}}
        Пример **запроса, меняющего сортировку**:

        %%(js)
        {
            "version": "10",
            "changes": [
                {
                    "soring_changed": {
                        "sorting": [  // в списке произвольное количество объектов
                            {"column1_name": "asc"}, // в объекте один ключ – имя столбца
                            {"column2_name": "desc"},  // значение "asc" или "desc"
                        ]
                    }
                }
            ]
        }
        %%

        Ответ:

        %%(js)
        {
            "data": {
                "success": true,
                "version": "10",
            }
        }
        %%

        {{a name="grid-change-select-column"}}
        Пример **запроса, изменяющего столбец типа Список/Множественный список**.

        В %%changes%% нужно поместить массив всех изменений столбца. Изменения применяются друг за
        другом. Массив изменений должен быть непустой. В каждом элементе массива должно быть не более одного изменения.

        Изменения бывают четырех типов:

        * %%{"add_option": {"value": "Сосиска"}}%% – добавить в конец списка элемент "Сосиска".
        * %%{"rename_option": {"index": 0, "new_value": "Борщ"}}%% – переименовать элемент с индексом 0 на "Борщ".
          При этом этот элемент будет переименован во всех строках грида соответствующий колонки.
        * %%{"remove_option": {"index": 1}}%% – удалить элемент с индексом 1.
          При этом этот элемент будет удален изо всех строк грида соответствующий колонки.
        * %%{"move_option": {"old_index": 1, "new_index": 3}}%% – поместить элемент с индексом 1 на место 3.
          Пример: %%[a, b, c, d, e] -> [a, c, d, b, e]%%.

        Индексы элементов начинаются с 0.

        %%(js)
        {
          "version": "43",
          "changes": [{
            "edited_column": {
              "name": "имя столбца",
              "options": {
                "changes": [
                  {"add_option": {"value": "Сырники"}},
                  {"add_option": {"value": "Отвага"}},
                  {"rename_option": {"index": 1, "new_value": "Крыжовник"}},
                  {"add_option": {"value": "Рогатина"}},
                  {"move_option": {"old_index": 2, "new_index": 0}},
                  {"remove_option": {"index": 1}}
                ]
              }
            }
          }]
        }
        %%

        Этот запрос соответствует следующей последовательности изменений списка опций:
        %%
        [] -> ["Сырники"] -> ["Сырники", "Отвага"] -> ["Сырники", "Крыжовник"] ->
        ["Сырники", "Крыжовник", "Рогатина"] -> ["Рогатина", "Сырники", "Крыжовник"] -> ["Рогатина", "Крыжовник"]
        %%

        Запрос, добавляющий столбец типа Список/Множественный список, составляется аналогично,
        нужно лишь добавить поля, необходимые для операции добавления столбца.

        ((#grid-change-top назад к списку))

        """
        try:
            self._grid = Grid.objects.select_for_update().get(id=request.page.id)
        except Grid.DoesNotExist:
            # Translators:
            #  ru: Табличный список не существует
            #  en: Grid does not exist
            raise Http404(_('Grid does not exist'))
        except OperationalError as err:
            if err.__cause__.pgcode == pg_errorcodes.LOCK_NOT_AVAILABLE:
                # не пятисотить, если не можем взять select_for_update у грида в течение lock_timeout и после ретраев.
                # WIKI-12087: фикс 500-к и ошибки в логах "OperationalError: canceling statement due to lock timeout"
                raise errors.ObjectIsLockedForUpdate()
            else:
                raise

        serializer = self.get_serializer(data=request.data)

        if serializer.is_valid():
            serializer.save(author_of_changes=self.request.user, grid=self._grid)
            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(self._grid)

            return Response(
                self.successful_response_serializer(
                    # successful_answer_proxy должен использоваться
                    # только в случае успешного ответа
                    self.successful_answer_proxy(self.request.user_auth, self._grid, success=True)
                ).data
            )
        else:
            raise InvalidDataSentError(non_field_messages=self.flatten_errors(serializer.errors))

    @staticmethod
    def flatten_errors(errors_dict):
        """
        Сделать ошибки плоскими

        Ошибки чаще всего вложенные вида
        {"changes":{'added_column': {'type': [u'This field is required.']}}}
        мы делаем их плоскими вида ['type: This field is required.']

        @type errors_dict: dict
        @rtype: list
        """
        result = []
        for field_name, message in errors_logic.yield_key_value_string_errors(errors_dict):
            result.append('{field_name}: {message}'.format(field_name=field_name, message=message))
        return result


class FullGridAnswerProxy(DefaultAnswerProxy):
    """
    Обертка для ручки
    """

    @property
    def grid(self):
        return GridProxy(self._grid, self._user_auth, {})


class ChangeGridAndGetItAllBackView(ChangeGridView):
    """
    Медленная ручка изменения табличного списка. Алиас к Change Grid.

    В успешном ответе на изменения грида присылает его весь обратно. Нужна,
    чтобы не приходилось поддерживать у себя актуальное состояние документа.

    """

    successful_response_serializer = FullGridVersionedStatusSerializer
    successful_answer_proxy = FullGridAnswerProxy
    check_readonly_mode = True

    @raises(
        errors.NoSuchRowError,
        errors.NoSuchColumnError,
        errors.CellHasChangedError,
        errors.ColumnHasChangedError,
        errors.TitleHasChangedError,
        errors.ConflictingChangeError,
        errors.SortingHasChangedError,
        errors.NoSuchOptionError,
        errors.DuplicateOptionError,
        errors.NoMoveTargetRowsError,
        errors.TicketDependentFieldWithoutTicket,
        errors.SortingHasChangedError,
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
        errors.ObjectIsLockedForUpdate,
    )
    def post(self, request, *args, **kwargs):
        """
        Изменить грид.

        Формат ответа и поведение такое же, как у быстрой аналогичной ручки.
        За исключением того, что в поле grid будет возвращен весь табличный
        список.
        """
        return super(ChangeGridAndGetItAllBackView, self).post(request, *args, **kwargs)


class ChangeCellAnswerProxy(object):
    _grid = None
    cell = None
    version = None

    message = None
    success = True

    def __init__(self, user_auth, grid, row_id, cell_id):
        """
        @type grid: Grid
        """
        cell_offset = grid.columns_meta().index(grid.column_by_name(cell_id))
        rows = GridProxy(grid, user_auth)._grid_rows(hashes=[row_id])
        self.cell = list(rows)[0][cell_offset]
        self._grid = grid

    @property
    def structure(self):
        structure = self._grid.access_structure.copy()

        # hack: разбить поле длинны с единицами измерения на два:
        # цифру и строку единиц измерения
        replace_width_in_grid_structure(structure)

        for field in structure['fields']:
            for sort in self.sorting:
                if field['name'] == sort['name']:
                    field['sorting'] = sort['type']
            else:
                if 'sorting' in field:
                    del field['sorting']
        return structure

    def __getattr__(self, item):
        return getattr(self._grid, item)


class ChangeCellAndGetBemJson(ChangeGridView):
    """
    Медленная ручка, в ответ на изменение текстовой ячейки отдает bemjson.

    В ручку можно присылать только изменение одной ячейки за раз, иначе она
    будет отдавать неконсистентные ответы.
    """

    successful_response_serializer = CellVersionedStatusSerializer
    successful_answer_proxy = ChangeCellAnswerProxy
    check_readonly_mode = True

    def get_cell_id(self, request, grid):
        """
        Вернуть пару (айди строки, имя столбца)

        @rtype: tuple
        """
        try:
            row_id = request.data['changes'][0]['edited_row']['id']
            data = request.data['changes'][0]['edited_row']['data']
            assert isinstance(data, dict)
            column_name = list(data.keys())[0]
        except (AssertionError, KeyError, IndexError) as exc:
            # возможно, неверный запрос от фронтэнда или эта функция отстала от
            # GridEditedRow - посмотрите в его код.
            raise InvalidDataSentError(
                # Translators:
                #  ru: Возможно, неверный запрос от фронтэнда, напишите %(email_contact)s, пожалуйста
                #  en: Probably a bad request from frontend, contact %(email_contact)s, please
                _('Probably a bad request from frontend, contact %(email_contact)s, please')
                % {'email_contact': get_support_contact()},
                debug_message=repr(exc),
            )

        return row_id, column_name

    @atomic_retry()
    @raises(
        errors.NoSuchRowError,
        errors.NoSuchColumnError,
        errors.NoSuchOptionError,
        errors.CellHasChangedError,
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
        errors.ObjectIsLockedForUpdate,
    )
    def post(self, request, *args, **kwargs):
        """
        Изменить ячейки табличного списка. То же самое можно получить, дергая
        ручку "Change Grid", но эта ручка отвечает еще и bemjson, который
        получился в результате изменения текстовой ячейки.
        """
        try:
            self._grid = Grid.objects.select_for_update().get(id=request.page.id)
        except Grid.DoesNotExist:
            # Translators:
            #  ru: Табличный список не существует
            #  en: Grid does not exist
            raise Http404(_('Grid does not exist'))
        except OperationalError as err:
            if err.__cause__.pgcode == pg_errorcodes.LOCK_NOT_AVAILABLE:
                # не пятисотить, если не можем взять select_for_update у грида в течение lock_timeout и после ретраев.
                # WIKI-12087: фикс 500-к и ошибки в логах "OperationalError: canceling statement due to lock timeout"
                raise errors.ObjectIsLockedForUpdate()
            else:
                raise

        serializer = self.get_serializer(data=request.data)

        if serializer.is_valid():
            # специально делается между serializer.is_valid() и serializer.save()
            row_id, column_name = self.get_cell_id(request, self._grid)

            serializer.save(author_of_changes=self.request.user, grid=self._grid)

            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(self._grid)
            return Response(
                self.successful_response_serializer(
                    # successful_answer_proxy должен использоваться
                    # только в случае успешного ответа
                    self.successful_answer_proxy(self.request.user_auth, self._grid, row_id, column_name)
                ).data
            )
        else:
            logger.warning(
                'Invalid request sent to grids "%d": "%s" (by "%s")',
                self._grid.id,
                serializer.errors,
                request.user.username,
            )
            raise InvalidDataSentError(serializer.errors)
