import logging

from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import paginated
from wiki.api_frontend.serializers.comments import CommentSerializer
from wiki.pages.access import is_admin
from wiki.pages.logic import comment as comment_logic
from wiki.pages.models import Comment
from wiki.pages.models.consts import COMMENTS_STATUS

logger = logging.getLogger(__name__)


class CommentsListView(PageAPIView):
    """
    View для работы со списком комментариев

    @todo: убрать CommentsArraySerializer, использовать встроенную
    пагинацию из django_rest_framework.
    """

    serializer_class = CommentSerializer

    DEFAULT_LIMIT = 100

    @raises()
    @paginated
    def get(self, request, start=None, limit=None, *args, **kwargs):
        """
        Получить список комментариев к странице.

        Получить 100 комментариев, начиная с сотого:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments?start=100&limit=100"
        %%

        %%(json)
        [
            {
                id: 16503,
                user: {...},
                parent: null,  // айдишник родительского комментария
                body: "asdasd",
                created_at: "2013-04-03T17:08:40",
                status: true,
                bemjson: {...}
            }
        ]
        %%

        """
        start = start or 0
        limit = limit or self.DEFAULT_LIMIT
        comments = Comment.objects.for_page(request.page).with_related()
        comments = comments.order_by('parent', 'id')
        comments = Comment.make_list_for_template(comments)
        comments = comments[start : start + limit]
        return self.build_response(instance=comments, context={'request': self.request}, many=True)

    @raises(InvalidDataSentError)
    def post(self, request, *args, **kwargs):
        """
        Создать комментарий к данной странице.

        Пример запроса, создающего ответный комментарий к комментарию с id 43:

        %%
        curl -H "Authorization: OAuth <token>" -XPOST \
        -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<supertag>/.comments" \
         --data '{"body": "Hi", "parent": 43}'
        %%

        %%(json)
        {
          "body": "Уильям Шетнер это не только Капитан Кирк, но и Денни Крейн",
          "parent": 43,
        }
        %%
        """
        ctx = {'request': self.request}
        data = self.validate(context=ctx, serializer_kwargs={'page': request.page})

        if request.page.comments_status != COMMENTS_STATUS.enabled:
            raise InvalidDataSentError(
                # Translators:
                #  ru: Возможность добавления комментариев на странице отключена
                #  en: The ability to add comments to the page are disabled
                _('The ability to add comments to the page are disabled'),
            )

        parent = data.get('parent')
        comment = comment_logic.add_comment(
            user=request.user,
            page=request.page,
            body=data['body'],
            parent_id=parent and parent.id,
        )
        return self.build_response(instance=comment, context=ctx)


class CommentDetailView(PageAPIView):
    """
    View для работы с отдельным комментарием.
    """

    serializer_class = CommentSerializer

    def initial(self, request, pk, *args, **kwargs):
        queryset = Comment.objects.for_page(self.request.page).active()
        queryset = queryset.with_related()
        self.comment = get_object_or_404(queryset, pk=pk)
        super(CommentDetailView, self).initial(request, *args, **kwargs)

    @raises()
    def get(self, request, pk, *args, **kwargs):
        """
        Вернуть комментарий по id.

        Пример запроса, получающего комментарий с id 43:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments/43"
        %%

        Ответ:

        %%(json)
        "data": {
            "id": 43,
            "user": {...},
            "parent": 10,
            "body": "hi",
            "created_at": "2013-04-03T17:08:40",
            "bemjson": {...}
        }
        %%
        """
        return self.build_response(instance=self.comment, context={'request': self.request})

    @raises()
    def post(self, request, pk, *args, **kwargs):
        """
        Редактирование комментария.

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -XPOST -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments/43" \
        --data '{"body": "wow"}'
        %%

        Успешный ответ будет со статусом 200.
        """
        ctx = {'request': self.request}
        data = self.validate(context=ctx, serializer_kwargs={'page': request.page})

        comment = comment_logic.edit_comment(
            comment=self.comment,
            user=request.user,
            body=data['body'],
        )
        return self.build_response(instance=comment, context=ctx)

    @raises()
    def delete(self, request, pk, *args, **kwargs):
        """
        Удаление комментария.

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -XDELETE -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments/43"
        %%

        Успешный ответ будет со статусом 200.
        """
        comment_logic.delete_comment(comment=self.comment, user=request.user)
        return self.build_success_status_response()

    def check_access_post(self):
        if is_admin(self.request.user):
            return

        if self.request.user != self.comment.user:
            raise UserHasNoAccess(_('Comment can be edited only by author'))

    def check_access_delete(self):
        request = self.request
        if is_admin(request.user):
            return
        if request.user not in request.page.get_authors() and request.user != self.comment.user:
            raise UserHasNoAccess(_('Comment can be deleted only by author or page owner'))
