import logging

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.comments import CommentsStatusDeserializer, CommentsStatusSerializer
from wiki.pages.logic import comment as comment_logic

logger = logging.getLogger(__name__)


class CommentsStatusView(PageAPIView):
    """
    View для работы со статусом доступности комментариев к странице.
    """

    serializer_class = CommentsStatusDeserializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить данные о статусе доступности комментариев к странице.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments/status"
        %%

        Тело ответа:

        %%(json)
        "data": {
            "comments_status": "enabled",  // "enabled" | "disabled_on_page" | "disabled_on_cluster"
            "subpages_count": 10,
        }
        %%
        """
        return self.build_response(instance=request.page, serializer_class=CommentsStatusSerializer)

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Изменить статус доступности комментариев у страницы или всего кластера.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
        -H -XPOST "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.comments/status" \
        --data '{"disabled": True, "for_cluster": False}'
        %%

        В случае отсутствия ошибок тело ответа будет пустым, а статус ответа будет 200.
        """
        data = self.validate()
        self.check_access_post()
        comment_logic.change_comments_status(request.page, disabled=data['disabled'], for_cluster=data['for_cluster'])
        return self.build_success_status_response()

    def check_page_access(self):
        pass

    def check_access_post(self):
        from wiki.pages import access

        if self.request.user not in self.request.page.get_authors() and not access.is_admin(self.request.user):
            raise UserHasNoAccess('Only page author can change page comments status')
