import logging

from django.db import transaction, IntegrityError
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.rest_api_error import ResourceAlreadyExists
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.grids import GridCreateSerializer
from wiki.api_frontend.serializers.page_view import PageCreationSerializer
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.grids.dao import create_grid
from wiki.pages.constants import ReservedCreationError

logger = logging.getLogger(__name__)


class CreateGridView(PageAPIView):
    """
    Позволяет создать новый пустой грид.
    """

    serializer_class = GridCreateSerializer

    def check_page_exists(self):
        pass

    @transaction.atomic
    @raises(ResourceAlreadyExists)
    def post(self, request, *args, **kwargs):
        """
        Создать грид.

        Пример запроса:

        %%(js)
        {
          "title": "some title"
        }
        %%

        Ответ:

        %%(js)
        {
          "data": {
              "success": true,
              "tag": "Тег",
              "supertag": "teg"
          }
        }
        %%
        """
        data = self.validate()

        if request.page:
            raise ResourceAlreadyExists()

        try:
            grid = create_grid(request.tag, data['title'], request.user)

            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(grid)
        except IntegrityError:
            raise ResourceAlreadyExists()
        except ReservedCreationError:
            raise InvalidDataSentError(_('Bad tag given: "{tag}"').format(tag=request.tag))

        return Response(
            PageCreationSerializer(
                {
                    'page_type': 'grid',
                    'tag': grid.tag,
                    'url': grid.url,
                    'supertag': grid.supertag,
                }
            ).data
        )
