import json
import logging

from django.http import Http404
from rest_framework.response import Response

from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.diff import GridDiffSerializer, PageDiffSerializer
from wiki.grids.logic import diff
from wiki.pages.models import Page, Revision
from wiki.utils.diff import DifflibDiff

logger = logging.getLogger(__name__)

_diff = DifflibDiff(return_type='chunks')


def _raw_content(field):
    if not field:
        return ''

    raw_content = field['raw']
    if isinstance(raw_content, bool):
        return '✓' if raw_content else ''
    elif isinstance(raw_content, list):
        return ', '.join(str(s) for s in raw_content)
    return str(raw_content)


class DiffView(PageAPIView):
    """
    Дифф страниц и гридов.
    """

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить diff страниц и табличных списков.

        Пример запроса, строящего diff между ревизиями 41 и 43:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.diff?a=41&b=43"
        %%

        %%(json)
        {data: {
            diff: [
                {
                    chunks: [
                        {
                            action: "deleted",
                            substring: "Володя обещал написать инструкцию, по которой я подключу conventionist к wf."
                        }
                    ],
                    changes: "deletions"
                },
                {
                    chunks: [
                        {
                            action: "deleted",
                            substring: ""
                        }
                    ],
                    changes: "deletions"
                },
                {
                    chunks: [
                        {
                            action: "not_changed",
                            substring: "Зачем делать WIKI-5923?"
                        }
                    ],
                    changes: "nothing"
                },
                {
                    chunks: [
                        {
                            action: "deleted",
                            substring: ""
                        }
                    ],
                    changes: "deletions"
                }
            ]}
        }
        %%

        Обязательные GET-параметры — a и b, равные id ревизий, между которыми
        требуется построить дифф. Порядок неважен, дифф будет всегда считаться
        хронологически от более старой к более новой ревизии.
        """
        rev_a_id, rev_b_id = self.request.GET.get('a'), self.request.GET.get('b')

        if not rev_a_id or not rev_b_id:
            raise Http404

        revisions = list(Revision.objects.filter(id__in=[rev_a_id, rev_b_id]).order_by('created_at'))
        if len(revisions) != 2:
            raise Http404

        old_rev, new_rev = revisions
        if old_rev.page_id != self.request.page.id:
            raise Http404

        if self.request.page.page_type == Page.TYPES.PAGE:
            diff = build_page_diff(old_rev, new_rev)
        elif self.request.page.page_type == Page.TYPES.GRID:
            diff = build_grid_diff(old_rev, new_rev)
        else:
            diff = {'diff': []}

        return Response(diff)


def build_page_diff(old_rev, new_rev):
    pages_diff = _diff.diff_texts(old_rev.body, new_rev.body)
    return PageDiffSerializer(pages_diff).data


def build_grid_diff(old_rev, new_rev):
    grids_diff, titles_diff = diff.build_diff(
        old_grid=json.loads(old_rev.body),
        new_grid=json.loads(new_rev.body),
    )
    return GridDiffSerializer(
        {
            'diff': grids_diff,
            'titles': titles_diff,
        }
    ).data
