import logging
from urllib.parse import unquote, urlparse

from django.conf import settings
from rest_framework.response import Response

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.explainlinks import ExplainLinksInputSerializer, ExplainLinksSerializer
from wiki.org import get_org
from wiki.pages.access import ACCESS_DENIED, get_bulk_access_status
from wiki.pages.models import Page, RedirectLoopException
from wiki.utils.path import canonize, resolve
from wiki.utils.supertag import tag_to_supertag

logger = logging.getLogger(__name__)


def explain(original_links, for_user):
    links = {}  # Ссылка -> супертег
    for supertag in original_links.keys():
        for link in original_links[supertag]:
            parsed_link = urlparse(link)
            if parsed_link.scheme and (
                parsed_link.scheme not in ('http', 'https') or parsed_link.netloc.lower() not in settings.FRONTEND_HOSTS
            ):
                continue  # Внешняя ссылка

            links[link] = tag_to_supertag(
                canonize(resolve('/' + supertag, unquote(parsed_link.path)))
            )

    # Нормализуем теги
    for link, supertag in links.items():
        # Убираем хэндлеры
        supertag_chunks = supertag.split('/')
        if supertag_chunks and supertag_chunks[-1].startswith('.'):
            supertag = '/'.join(supertag_chunks[:-1])

        # Супертег главной страницы
        if not supertag:
            supertag = settings.MAIN_PAGE

        links[link] = supertag

    # найти редиректы
    pages = Page.objects.filter(supertag__in=set(links.values()), org=get_org())
    redirects = {}  # Супертег -> супертег
    for page in pages:
        if page.has_redirect():
            try:
                redirects[page.supertag] = page.redirect_target().supertag
            except RedirectLoopException:
                pass

    access_status = get_bulk_access_status(list(links.values()), for_user)
    redirect_access_status = get_bulk_access_status(list(redirects.values()), for_user)

    result = {}
    for link, supertag in links.items():
        status = access_status[supertag]
        if status > ACCESS_DENIED and supertag in redirects:
            # Есть доступ до редиректа, надо вернуть статус ссылаемой страницы
            if redirects[supertag] in redirect_access_status:
                status = redirect_access_status[redirects[supertag]]
        result[link] = status

    return result


class ExplainLinksView(WikiAPIView):
    """
    Права пользователя на страницы.
    """

    serializer_class = ExplainLinksInputSerializer

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Позволяет получить права пользователя на список страниц по ссылкам. Ручка не завязана на конкретную страницу.

        Вход (POST):
        %%(js)
            {
                "links": {
                    "wiki/dev": ["/Портос/Арамис", "http://wiki.yandex-team.ru/Wiki/Vodstvo",],
                    "wiki/Vodstvo": ["Атос", "http://ya.ru/"],
                }
            }
        %%

        Выход:
        %%(js)
            {
                "links": {
                    "Атос": 1,
                    "/Портос/Арамис": 2,
                    "http://wiki.yandex-team.ru/Wiki/Vodstvo": 3
                }
            }
        %%

        В качестве ключей выступают ссылки, они могут быть тегами (опционально - со слешами на концах).
        Также допускаются полные ссылки на вики: http://wiki.yandex-team.ru/Страница.
        Ссылки могут заканчиваться на /.show, тогда это окончание игнорируется.
        Ссылки не на вики отсутствуют в ответе. Все остальные ссылки (в том числе на несуществующие страницы в вики)
        присутствуют в ответе в том же виде, в котором были отправлены.

        Значения — константы из wiki.pages.acccess.access_status:

            ACCESS_NONEXISTENT = -1
            ACCESS_DENIED = 0
            ACCESS_COMMON = 1
            ACCESS_RESTRICTED = 2
            ACCESS_UNLIMITED = 3

        """
        data = self.validate()
        result = explain(data['links'], request.user)
        return Response(ExplainLinksSerializer({'links': result}).data)
