
from rest_framework.response import Response

from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.export import ExportViewParams
from wiki.async_process.backend.consts import AsyncTaskType
from wiki.async_process.tasks.async_request_processor import delay_async_request
from wiki.pages.logic.export.utils import get_result_file_name_without_ext


class ExportView(PageAPIView):
    serializer_class = ExportViewParams

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Экспортировать вики-страницу.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/Godzilla/Kurilla/.export?ext=pdf&with_subpages=True"
        %%

        Принимает следующие GET-параметры:

        * ext: 'pdf' | 'docx' – формат, в который экспортируем вики-страницу,
        * with_subpages: true | false – экспортируем одну страницу либо весь кластер.

        """
        get_params = self.validate(data=request.GET)

        # такие запросы не должны быть синхронными.
        # чем меньше занят api gateway тем стабильнее сервис
        # поэтому убрал логику на то что иногда мы отдаем синхронно

        # @todo - переделать схему на POST

        return delay_async_request(
            task_type=AsyncTaskType.EXPORT,
            task_data={
                'page_id': request.page.id,
                'output_format': get_params['ext'],
                'with_subpages': get_params.get('with_subpages', False),
                'user_id': request.user.id,
                'cookies': kwargs.get('cookies'),
            },
        )


class ExportFilenameView(PageAPIView):
    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить имя для экспортируемого файла без расширения.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/Godzilla/Kurilla/.export_filename"
        %%

        Пример ответа:

        {
            "filename": "chaoscucumber33"
        }

        """
        return Response({'filename': get_result_file_name_without_ext(request)})
