import logging

from django.db import transaction
from django.db.models import F
from django.http import Http404
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.favorites import (
    BookmarkEditionInputSerializer,
    BookmarkInputSerializer,
    BookmarkListModelSerializer,
    BookmarkListSerializer,
    BookmarkModelSerializer,
    BookmarkMoveSerializer,
    BookmarkSerializer,
    LinkedBookmarksDeletionSerializer,
)
from wiki.api_frontend.serializers.io import ok_response
from wiki.favorites_v2.dao import (
    create_bookmark,
    create_bookmark_compat,
    get_or_create_folder,
    get_user_bookmarks_by_folders,
    get_user_bookmarks_by_folders_compat,
    update_bookmarks_indexes,
)
from wiki.favorites_v2.models import Bookmark, Folder
from wiki.favorites.logic import update_bookmark_tags
from wiki.favorites.models import Bookmark as NewBookmark
from wiki.org import get_org
from wiki.sync.connect.base_organization import as_base_organization
from wiki.users.logic.settings import uses_new_favorites

logger = logging.getLogger(__name__)


class BookmarkCreateView(WikiAPIView):
    """
    View для работы с закладкой: создание, удаление по супертегу, получение списка закладок.
    """

    render_blank_form_for_methods = ('PUT',)

    def get_serializer_class(self):
        if self.request.method == 'PUT':
            return BookmarkInputSerializer
        elif self.request.method == 'POST':
            return LinkedBookmarksDeletionSerializer
        else:
            return BookmarkListModelSerializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить список всех закладок пользователя во всех папках (включая автопапки).
        Закладки группируются по папкам.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks"
        %%

        Пример ответа:

        %%(json)
        {
            "debug": {
                ....                            // дебаг информация, в продакшне может быть отключена
            },
            "data": [
                {
                    "folder_name": "__INBOX__",
                    "bookmarks": [
                        {
                            "id": ​​69967,
                            "title": "Руководство по сервису Вики",
                            "url": "https://wiki.yandex-team.ru/wiki/vodstvo",
                            "page_modified_at": "2015-01-09T08:12:06",
                            "page_last_editor": "techpriest"
                        },
                        ...
                    ]
                },
                ...
            ]
            "user": {
                ....
            }
        }
        %%
        """
        getting_bookmarks_function = get_user_bookmarks_by_folders
        serializer = BookmarkListModelSerializer
        if uses_new_favorites(request.user):
            getting_bookmarks_function = get_user_bookmarks_by_folders_compat
            serializer = BookmarkListSerializer

        data = [
            {'folder_name': key, 'bookmarks': value}
            for key, value in list(getting_bookmarks_function(request.user).items())
        ]

        return Response(serializer(data, many=True).data)

    @raises()
    @transaction.atomic
    def put(self, request, *args, **kwargs):
        """
        Создать закладку в папке с указанным именем в Избранном пользователя.
        Невозможно создать закладку в автопапке.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%folder_name%% | str | !!обязательный!! | имя папки для закладки ||
        || %%title%% | str | !!обязательный!! | название закладки ||
        || %%url%% | str | !!обязательный!! | ссылка на страницу закладки ||
        |#

        Возвращает: json созданной закладки аналогично GET-запросу. Например:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "PUT" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks" \
        --data 'тело запроса'
        %%

        тело запроса:
        %%(json)
        {
            "title": "текущие задачи форматтера"
            "url": ""http://wiki.yandex-team.ru/Users/yurgis/Projects/WikiWt2Html/Flow/CurrentTasks/.show",
            "folder_name": "Избранное"
        }
        %%

        Пример ответа:
        %%(js)
        {
            "data": {
                "id": 156466,
                "title": "текущие   задачи форматтера",
                "url": "http://wiki.yandex-team.ru/Users/yurgis/Projects/WikiWt2Html/Flow/CurrentTasks/.show",
                "page_modified_at": "2013-04-03T17:08:40+03:00",
                "page_last_editor": "yurgis"
            }
        }
        %%

        Если папка с указанным именем не существует, то она будет создана.

        Если обязательные параметры %%title%% и %%url%% отсутствуют в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "url": ["This field is required."],
                    "title": ["This field is required."]
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        При попытке создать закладку в автопапке в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Сan't create bookmark in this folder.",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        data = self.validate()
        folder_name = data.get('folder_name')
        if folder_name in Folder.AUTOFOLDERS:
            # Translators:
            #  ru: Нельзя создать закладку в этой папке
            #  en: Сan't create bookmark in this folder.
            raise InvalidDataSentError(_('Сan\'t create bookmark in this folder.'))

        serializer = BookmarkModelSerializer
        if uses_new_favorites(request.user):
            serializer = BookmarkSerializer
            bookmark = create_bookmark_compat(request.user, data.get('url'), folder_name)
        else:
            folder = get_or_create_folder(request.user, folder_name)
            bookmark = create_bookmark(folder, data.get('title'), data.get('url'))

        return Response(serializer(bookmark).data)

    @raises()
    @transaction.atomic
    def post(self, request, *args, **kwargs):
        """
        Удалить закладки, указывающие на вики страницу с указанным супертегом.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%supertag%% | str | !!обязательный!! | супертег вики страницы ||
        |#

        Возвращает: статус-код ответа.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks" \
        --data '{"supertag": "testpage"}'
        %%

        Если обязательный параметр %%supertag%% отсутствует в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "supertag": ["This field is required."]
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        200ый ответ говорит об успешном окончании операции.

        """
        data = self.validate()
        supertag = data['supertag']
        if uses_new_favorites(request.user):
            bookmarks = NewBookmark.objects.filter(
                user=request.user,
                page__org=get_org(),
                page__supertag=supertag
            ).delete()
        else:
            bookmarks = Bookmark.objects.select_related('folder').filter(
                supertag=supertag, folder__user=request.user, folder__org=get_org()
            )

            Folder.objects.filter(id__in=[bookmark.folder_id for bookmark in bookmarks]).update(
                favorites_count=F('favorites_count') - 1
            )
            bookmarks.delete()

        return ok_response()


class BookmarkEditView(WikiAPIView):
    """
    View для работы с закладкой: редактирование, перемещение и удаление.
    """

    serializer_class = BookmarkEditionInputSerializer

    @raises()
    def post(self, request, bookmark_id, *args, **kwargs):
        """
        Редактировать закладку.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%title%% | str | !!обязательный!! | новое название закладки ||
        |#

        Возвращает: json закладки аналогично GET-запросу. Например:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks/156466" \
        --data '{тело запроса}'
        %%

        тело запроса
        %%(json)
        {
          "title": "текущие задачи форматтера",
          "url": "http://wiki.yandex-team.ru/Users/yurgis/Projects/.show",
        }
        %%

        Пример ответа:
        %%(json)
        {
            "data": {
                "id": 156466,
                "title": "текущие задачи форматтера",
                "url": "http://wiki.yandex-team.ru/Users/yurgis/Projects/.show",
                "page_modified_at": "2013-11-08T22:46:00",
                "page_last_editor": "yurgis"
            }
        }
        %%

        Если обязательный параметр %%title%% отсутствует в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(json)
        {
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "title": ["This field is required."]
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если закладка с указанным ID не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "error": {
                "message": "Http404(u'Bookmark id=10101 not found',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        При попытке редактировать закладку в автопапке, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Сan't edit bookmark in this folder",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        if uses_new_favorites(self.request.user):
            # В новой схеме title закладки == title страницы
            raise Http404

        try:
            bookmark = Bookmark.objects.select_related('folder').get(id=bookmark_id)
        except Bookmark.DoesNotExist:
            raise Http404

        data = self.validate()

        if bookmark.folder.type == Folder.FOLDER_TYPE_AUTO:
            # Translators:
            #  ru: Нельзя редактировать закладку в этой папке
            #  en: Сan't edit bookmark in this folder.
            raise InvalidDataSentError(_('Сan\'t edit bookmark in this folder.'))

        bookmark.title = data['title']
        bookmark.save()
        return Response(BookmarkModelSerializer(bookmark).data)

    @raises()
    @transaction.atomic
    def delete(self, request, bookmark_id, *args, **kwargs):
        """
        Удалить закладку.

        Возвращает: статус-код ответа.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "DELETE" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks/43"
        %%

        Если закладка с указанным ID не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(json)
        {
            "error": {
                "message": "Http404(u'Bookmark id=10101 not found',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        При попытке удалить закладку из автопапки, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(json)
        {
            "error": {
                "message": "Сan't delete bookmark in this folder.",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        if uses_new_favorites(self.request.user):
            bookmark = get_object_or_404(NewBookmark, id=bookmark_id, user=self.request.user, page__org=get_org())
            bookmark.delete()
        else:
            try:
                bookmark = Bookmark.objects.select_related('folder').get(
                    id=bookmark_id,
                    folder__user=self.request.user,
                    folder__org=get_org()
                )
                folder = bookmark.folder
                if bookmark.folder.type == Folder.FOLDER_TYPE_AUTO:
                    # Translators:
                    #  ru: Нельзя удалить закладку в этой папке
                    #  en: Сan't delete bookmark in this folder.
                    raise InvalidDataSentError(_('Сan\'t delete bookmark in this folder.'))

                folder.favorites_count -= 1
                folder.save()

                bookmark.delete()
            except Bookmark.DoesNotExist:
                raise Http404

        return ok_response()


class MoveBookmarkView(WikiAPIView):
    """
    View для работы с закладкой: перемещение из одной папки в другую.
    """

    serializer_class = BookmarkMoveSerializer

    @staticmethod
    @transaction.atomic
    def move_bookmark(bookmark, target_folder):
        """
        Переместить закладку %%bookmark%% из текущей папки в %%target_folder%%.
        """
        folder = bookmark.folder
        if folder.type == Folder.FOLDER_TYPE_AUTO:
            # Translators:
            #  ru: Нельзя перемещать закладку из этой папки
            #  en: Сannot move bookmark from this folder.
            raise InvalidDataSentError(_('Сan\'t move bookmark from this folder.'))

        if target_folder.type == Folder.FOLDER_TYPE_AUTO:
            # Translators:
            #  ru: Нельзя перемещать закладку в эту папку
            #  en: Сannot move bookmark to this folder.
            raise InvalidDataSentError(_('Сan\'t move bookmark to this folder.'))

        bookmark.folder = target_folder
        bookmark.index_number = 0 if target_folder.favorites_count else 1
        bookmark.save()

        folder.favorites_count -= 1
        folder.save()

        target_folder.favorites_count += 1
        target_folder.save()

        if target_folder.favorites_count > 1:
            update_bookmarks_indexes(target_folder)

        return ok_response()

    @raises()
    @transaction.atomic
    def post(self, request, bookmark_id, *args, **kwargs):
        """
        Переместить закладку из текущей папки в другую.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%target_folder_name%% | str | !!обязательный!! | имя папки назначения ||
        |#

        Возвращает: статус-код ответа.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks/43/move" \
        --data 'тело запроса'
        %%

        тело запроса:
        %%(js)
        {
            "target_folder_name": "текущие задачи форматтера"                 // имя папки назначения
        }
        %%

        Если папка с переданным именем не существует, то она будет создана.

        Если обязательный параметр %%target_folder_name%% отсутствует в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "name": ["This field is required."]
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если закладка с указанным ID не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Http404(u'Bookmark id=10101 not found',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        При попытке переместить закладку из автопапки, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Сan't move bookmark from this folder.",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        При попытке переместить закладку в автопапку, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Сan't move bookmark to this folder.",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        data = self.validate()
        target_folder_name = data['target_folder_name']

        if uses_new_favorites(self.request.user):
            if target_folder_name in Folder.AUTOFOLDERS:
                raise InvalidDataSentError(_('Сan\'t move bookmark to this folder.'))

            bookmark = get_object_or_404(NewBookmark, id=bookmark_id, user=self.request.user, page__org=get_org())
            tags = [] if target_folder_name in Folder.RESERVED_FOLDER_NAMES_LIST else [target_folder_name]
            update_bookmark_tags(self.request.user, as_base_organization(get_org()), bookmark, tags)

        else:
            target_folder = get_or_create_folder(request.user, target_folder_name)

            try:
                bookmark = Bookmark.objects.select_related('folder').get(id=bookmark_id)
            except Bookmark.DoesNotExist:
                raise Http404()

            self.move_bookmark(bookmark, target_folder)

        return ok_response()


class ChangeBookmarkOrderView(WikiAPIView):
    """
    View для работы с закладкой: изменение порядка отображения.
    """

    @raises()
    @transaction.atomic
    def post(self, request, bookmark_id, *args, **kwargs):
        """
        Изменить местоположение закладки относительно других закладок в текущей папке пользователя.

        Возвращает: статус-код ответа.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%after%% | str | не обязательный | id закладки, после которой теперь должна отображаться
        перетаскиваемая закладка. Если параметр отсутствует, то закладка перемещается на первое место ||
        |#

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "DELETE" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/bookmarks/43/drag" \
        --data '{тело запроса}'
        %%

        тело запроса:
        %%(js)
        {
            "after": "12"            // id закладки, после которой теперь должна отображаться перетаскиваемая закладка
        }
        %%

        Если закладка с указанным id не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "error": {
                "message": "Http404(u'Bookmark id=10101 not found',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        Если параметр %%after%% содержит id закладки, которой нет у данного пользователя,
        то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "error": {
                "message": 'No such Bookmark with id="10101"',
                "errors": {
                    'after': 'This field is required.'
                },
                "error_code": "NOT_FOUND"
            }
        }
        %%

        Если параметр %%after%% содержит id закладки, идентичное значению id перетаскиваемой закладки,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'after': 'This field can't be the same as the target bookmark id'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если параметр %%after%% содержит id закладки из другой папки, чем у перетаскиваемой закладки,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'after': 'This field contains bookmark id from the another Folder than the target bookmark'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        При попытке изменить местоположение закладки в автопапке, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Сan't move bookmark in this folder.",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        if uses_new_favorites(self.request.user):
            # В новой схеме не поддерживаем изменение порядка
            return ok_response()

        try:
            bookmark = Bookmark.objects.get(id=bookmark_id)
        except Bookmark.DoesNotExist:
            raise Http404('Bookmark id={0} not found'.format(bookmark_id))

        if bookmark.folder.type == Folder.FOLDER_TYPE_AUTO:
            # Translators:
            #  ru: Нельзя перемещать закладку в эту папку
            #  en: Сannot move bookmark in this folder.
            raise InvalidDataSentError(_('Сan\'t move bookmark in this folder.'))

        after_bookmark_id = self.request.data.get('after')

        if after_bookmark_id:
            if int(bookmark_id) == int(after_bookmark_id):
                # явно ошибочный запрос
                # Translators:
                #  ru: Нельзя перемещать закладку после себя самой
                #  en: You may not move bookmark after itself
                raise InvalidDataSentError(_('"after" can\'t be the same as the target bookmark id'))

            try:
                after_bookmark = Bookmark.objects.get(id=after_bookmark_id)
                if after_bookmark.folder_id != bookmark.folder_id:
                    # закладки из разных папок
                    raise InvalidDataSentError(
                        # Translators:
                        #  ru: Закладки из разных папок
                        #  en: The bookmarks are from different folders
                        _('This field contains bookmark id from the another Folder than the target bookmark')
                    )

                start_index_number = after_bookmark.index_number
            except Bookmark.DoesNotExist:
                raise Http404('Bookmark id={0} not found'.format(bookmark_id))
        else:
            start_index_number = 0

        qs = Bookmark.objects.filter(folder_id=bookmark.folder_id, index_number__gt=start_index_number)

        if start_index_number < bookmark.index_number:
            qs.filter(index_number__lt=bookmark.index_number)

        qs.update(index_number=F('index_number') + 1)
        bookmark.index_number = start_index_number + 1
        bookmark.save()

        return ok_response()
