import logging

from django.db import transaction
from django.db.models import Count, F
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.rest_api_error import ResourceAlreadyExists
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.favorites import (
    BookmarkSerializer,
    BookmarkModelSerializer,
    FolderMoveSerializer,
    FolderRenameSerializer,
    FolderSerializer,
    FolderModelSerializer,
)
from wiki.api_frontend.serializers.io import ok_response
from wiki.favorites_v2.dao import get_folder, get_or_create_folder
from wiki.favorites_v2.models import AutoBookmark, Bookmark, Folder
from wiki.favorites.consts import AutoBookmarkType
from wiki.favorites.models import AutoBookmark as NewAutoBookmark, Bookmark as NewBookmark, Tag
from wiki.org import get_org
from wiki.users.logic.settings import uses_new_favorites

logger = logging.getLogger(__name__)


class FoldersListView(WikiAPIView):
    """
    View для получения списка папок из Избранного пользователя.
    """

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить список папок из Избранного пользователя.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders"
        %%

        Пример ответа:

        %%(json)
        {
            "debug": {
                ....                                        // дебаг информация, в продакшне может быть отключена
            },
            "data": [
                {
                    "name": "__INBOX__",
                    "favorites_count": 10,
                    "type": "F"
                },
                {
                    "name": "Инструменты",
                    "favorites_count": 8,
                    "type": "C"
                },
                {
                    "name": "Вики",
                    "favorites_count": 14,
                    "type": "C"
                },
                {
                    "name": "Технологические штуки",
                    "favorites_count": 7,
                    "type": "C"
                }
            ],
        }
        %%

        """
        org = get_org()
        if uses_new_favorites(self.request.user):
            inbox_count = NewBookmark.objects.filter(user=self.request.user, page__org=org, tags=None).count()
            data = [{
                'name': Folder.FAVORITES_FOLDER_NAME,
                'favorites_count': inbox_count,
                'type': Folder.FOLDER_TYPE_FAVORITES,
            }]

            tag_bookmarks = (
                Tag.objects.annotate(bookmarks_count=Count('bookmark'))
                .filter(user=self.request.user, org=org).values('name', 'bookmarks_count')
            )
            count_by_tag = {item['name']: item['bookmarks_count'] for item in tag_bookmarks}

            tags = Tag.objects.filter(user=self.request.user, org=org).order_by('name')
            for tag in tags:
                data.append({
                    'name': tag.name,
                    'favorites_count': count_by_tag.get(tag.name, 0),
                    'type': Folder.FOLDER_TYPE_CUSTOM,
                })

            watcher_count = AutoBookmark.objects.filter(
                folder__user=self.request.user,
                folder__org=org,
                folder__name=Folder.WATCHER_AUTOFOLDER_NAME,
            ).count()

            owner_count = NewAutoBookmark.objects.filter(
                user=self.request.user,
                page__org=org,
                bookmark_type=AutoBookmarkType.CREATOR,
            ).count()

            editor_count = NewAutoBookmark.objects.filter(
                user=self.request.user,
                page__org=org,
                bookmark_type=AutoBookmarkType.EDITOR,
            ).count()

            data.extend([
                {
                    'name': Folder.OWNER_AUTOFOLDER_NAME,
                    'favorites_count': owner_count,
                    'type': Folder.FOLDER_TYPE_AUTO,
                },
                {
                    'name': Folder.WATCHER_AUTOFOLDER_NAME,
                    'favorites_count': watcher_count,
                    'type': Folder.FOLDER_TYPE_AUTO,
                },
                {
                    'name': Folder.LAST_EDIT_AUTOFOLDER_NAME,
                    'favorites_count': editor_count,
                    'type': Folder.FOLDER_TYPE_AUTO,
                },
            ])
        else:
            data = Folder.objects.filter(user=self.request.user, org=org).order_by('index_number')
        return Response(FolderModelSerializer(data, many=True).data)


def folder_by_name_and_user(folder_name, user):
    try:
        return Folder.objects.filter(user=user, name=folder_name, org=get_org())
    except Folder.DoesNotExist:
        return None


class FolderView(WikiAPIView):
    """
    View для работы с папкой: создание, редактирование и удаление.
    """

    def get_serializer_class(self):
        if self.request.method == 'PUT':
            from rest_framework.serializers import Serializer

            return Serializer
        elif self.request.method == 'POST':
            return FolderRenameSerializer

    @raises()
    def get(self, request, folder_name, *args, **kwargs):
        """
        Получить список закладок из указанной папки пользователя в Избранном.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders/<имяпапки>"
        %%

        Пример ответа:

        %%(json)
        {
            "data": [
                {
                    "id": 156466,
                    "title": "текущие задачи форматтера",
                    "url": "http://wiki.yandex-team.ru/Users/yurgis/Projects/WikiWt2Html/Flow/CurrentTasks/.show",
                    "page_modified_at": "2013-04-03T17:08:40+03:00",
                    "page_last_editor": "yurgis"
                },
                {
                    "id": 156467,
                    "title": "Серверы вики",
                    "url": "http://wiki.yandex-team.ru/wiki/dev/servers",
                    "page_modified_at": "2013-04-03T17:08:40+03:00",
                    "page_last_editor": "elisei"
                },
                {
                    "id": 156486,
                    "title": "Графики",
                    "url": "http://mould.yandex.ru/cacti/graph_view.php?action=tree&tree_id=50&leaf_id=2393",
                    "page_modified_at": null,
                    "page_last_editor": null
                },
            ],
        }
        %%

        """
        serializer = BookmarkModelSerializer
        if uses_new_favorites(self.request.user) and folder_name != Folder.WATCHER_AUTOFOLDER_NAME:
            serializer = BookmarkSerializer
            if folder_name == Folder.FAVORITES_FOLDER_NAME:
                # В новой схеме закладкам из папки избранного соответсвуют закладкам без тегов
                data = NewBookmark.objects.filter(
                    user=self.request.user,
                    page__org=get_org(),
                    tags=None
                ).order_by('-created_at')
            elif folder_name in [Folder.OWNER_AUTOFOLDER_NAME, Folder.LAST_EDIT_AUTOFOLDER_NAME]:
                bookmark_type = (
                    AutoBookmarkType.CREATOR if folder_name == Folder.OWNER_AUTOFOLDER_NAME else AutoBookmarkType.EDITOR
                )
                data = NewAutoBookmark.objects.filter(
                    user=self.request.user,
                    page__org=get_org(),
                    bookmark_type=bookmark_type
                ).order_by('-created_at')
            else:  # Пользовательская папка
                data = NewBookmark.objects.filter(
                    tags__name=folder_name,
                    tags__user=self.request.user,
                    tags__org=get_org(),
                ).order_by('-created_at')
        else:
            folder = get_object_or_404(Folder, user=self.request.user, name=folder_name, org=get_org())

            if folder.type == Folder.FOLDER_TYPE_AUTO:
                data = AutoBookmark.objects.filter(folder_id=folder.id).order_by('-page_modified_at')
            else:
                data = Bookmark.objects.filter(folder=folder).order_by('index_number')

        return Response(serializer(data, many=True).data)

    @raises(ResourceAlreadyExists)
    def put(self, request, folder_name, *args, **kwargs):
        """
        Создать папку с указанным именем в Избранном пользователя.

        Возвращает: json созданной папки аналогично GET-запросу. Например:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "PUT" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders/<имяпапки>" \
        --data '{тело запроса}'
        %%

        Пример ответа:
        %%(js)
        {
            "data": {
                "name": "Ссылки на Вики",
                "favorites_count": 10,
                "type": "C"
            }
        }
        %%

        Если папка с указанным именем уже существует, то в ответе будет передана ошибка с кодом 'ALREADY_EXISTS':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Resource already exists",
                "error_code": "ALREADY_EXISTS"
            }
        }
        %%

        Если пользователь попытается создать папку с именем, совпадающим с одним из зарезервированных имен для папок
        типа "Избранное" или автопапки, то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "name": "This name is reserved for special Folder"
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        self.validate()
        serializer = FolderModelSerializer
        if uses_new_favorites(request.user):
            serializer = FolderSerializer
            already_exists = (
                folder_name in Folder.RESERVED_FOLDER_NAMES_LIST
                or Tag.objects.filter(user=request.user, name=folder_name, org=get_org()).exists()
            )
            if already_exists:
                raise ResourceAlreadyExists
            tag = Tag.objects.create(user=request.user, name=folder_name, org=get_org())
            folder = {
                'name': tag.name,
                'favorites_count': 0,
                'type': Folder.FOLDER_TYPE_CUSTOM,
            }
        else:
            if get_folder(folder_name, request.user):
                raise ResourceAlreadyExists

            folder = get_or_create_folder(request.user, folder_name)

        return Response(serializer(folder).data)

    @raises(ResourceAlreadyExists)
    def post(self, request, folder_name, *args, **kwargs):
        """
        Отредактировать папку с указанным именем в Избранном пользователя.
        Допускается редактирование только пользовательских папок.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%new_name%% | str | !!обязательный!! | новое значение имени папки ||
        |#

        Возвращает json созданной папки аналогично GET-запросу. Например:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders/<имяпапки>" \
        --data '{тело запроса}'
        %%

        тело запроса:
        %%(js)
        {
            "new_name": "Все важные ссылки"                              // новое имя папки
        }
        %%

        Пример ответа:
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "name": "Все важные ссылки",
                "favorites_count": 10,
                "type": "C"
            }
        }
        %%

        Если папка с указанным именем не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Http404('No Folder matches the given query.',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        Если папка с новым именем уже существует, то в ответе будет передана ошибка с кодом 'ALREADY_EXISTS':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Resource already exists",
                "error_code": "ALREADY_EXISTS"
            }
        }
        %%

        Если обязательный параметр %%new_name%% отсутствует в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    "name": ["This field is required."]
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если редактируемая папка не является пользовательской, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "This folder can't be modified",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        if folder_name in Folder.RESERVED_FOLDER_NAMES_LIST:
            # Translators:
            #  ru: Эту папку нельзя модифицировать
            #  en: This folder can't be modified
            raise InvalidDataSentError(_('This folder can\'t be modified'))

        data = self.validate()
        serializer = FolderModelSerializer
        if uses_new_favorites(request.user):
            serializer = FolderSerializer
            tag = get_object_or_404(Tag, user=request.user, name=folder_name, org=get_org())

            new_name = data['new_name']
            if Tag.objects.filter(user=request.user, name=new_name, org=get_org()).exists():
                raise ResourceAlreadyExists

            tag.name = new_name
            tag.save()
            folder = {
                'name': tag.name,
                'favorites_count': tag.bookmark_set.count(),
                'type': Folder.FOLDER_TYPE_CUSTOM,
            }
        else:
            folder = get_object_or_404(Folder, user=request.user, name=folder_name, org=get_org())
            if get_folder(data['new_name'], request.user):
                raise ResourceAlreadyExists
            folder.name = data['new_name']
            folder.save()
        return Response(serializer(folder).data)

    @raises()
    @transaction.atomic
    def delete(self, request, folder_name, *args, **kwargs):
        """
        Удалить папку с закладками. Закладки из удаляемой папки перенести в папку %%Избранное%%.
        Удалять можно только пользовательские папки.

        Возвращает: статус-код ответа.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "DELETE" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders/<имяпапки>"
        %%

        Если папка с указанным именем не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Http404('No Folder matches the given query.',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        При попытке удалить папку, не являющуюся пользовательской, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "This folder can't be deleted",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        if folder_name in Folder.RESERVED_FOLDER_NAMES_LIST:
            # Translators:
            #  ru: Эту папку нельзя удалять
            #  en: This folder can't be deleted
            raise InvalidDataSentError(_('This folder can\'t be deleted'))

        if uses_new_favorites(self.request.user):
            tag = get_object_or_404(Tag, user=request.user, name=folder_name, org=get_org())
            tag.delete()
        else:
            folder = get_object_or_404(Folder, user=self.request.user, name=folder_name, org=get_org())

            bookmarks = Bookmark.objects.filter(folder=folder).order_by('-index_number').select_related('folder')

            target_folder = get_object_or_404(
                Folder, user=self.request.user, name=Folder.FAVORITES_FOLDER_NAME, org=get_org()
            )
            for bookmark in bookmarks:
                # импортируем здесь, чтобы избежать циклического иморта
                from wiki.api_frontend.views.favorites_bookmarks import MoveBookmarkView

                MoveBookmarkView.move_bookmark(bookmark, target_folder)

            folder.delete()

        return ok_response()


class ChangeFolderOrderView(WikiAPIView):
    """
    View для работы с папкой: изменение порядка отображения.
    """

    serializer_class = FolderMoveSerializer

    @raises()
    @transaction.atomic
    def post(self, request, folder_name, *args, **kwargs):
        """
        Изменить местоположение папки относительно других папок пользователя.
        Возможно только для пользовательских папок.

        Возвращает: статус-код ответа.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%after%% | str | !!обязательный!! | имя папки, после которой теперь должна отображаться
        перетаскиваемая папка ||
        |#

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.favorites/folders/<folder name>/drag" \
        --data 'тело запроса'
        %%

        тело запроса:
        %%(json)
        {
            "after": "My folder"            // имя папки после которой теперь должна отображаться перетаскиваемая папка
        }
        %%

        Если папка с указанным именем не существует, то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Http404('No Folder matches the given query.',)",
                "error_code": "NOT_FOUND"
            }
        }
        %%

        При попытке переместить папку, не являющуюся пользовательской, в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "This folder can't be moved",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если обязательный параметр %%after%% отсутствует в теле запроса, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": "Client sent invalid data",
                "errors": {
                    'after': 'This field is required.'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если параметр %%after%% содержит имя папки, которой нет у данного пользователя,
        то в ответе будет передана ошибка с кодом 'NOT_FOUND':
        %%(json)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": 'No such Folder with name "unknown" in user "thasonic"',
                "errors": {
                    'after': 'This field is required.'
                },
                "error_code": "NOT_FOUND"
            }
        }
        %%

        Если параметр %%after%% содержит имя папки, идентичное имени перетаскиваемой папки,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'after': 'This field can't be the same as the target folder'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Переместить пользовательскую папку на место после автопапки нельзя. При попытке осуществить такую операцию
        в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "debug": {
                ....                        // дебаг информация, в продакшне может быть отключена
            },
            "error": {
                "message": 'The folder can\'t be moved after required folder',
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        """
        data = self.validate()
        after_folder_name = data['after']

        if uses_new_favorites(self.request.user):
            # В новой схеме не поддерживаем изменение порядка
            return ok_response()

        if not folder_name:
            # Translators:
            #  ru: Поле "имя папки" обязательно.
            #  en: Folder name is required.
            raise InvalidDataSentError({'folder_name': _('Folder name is required.')})

        if after_folder_name == folder_name:
            # явно ошибочный запрос
            # Translators:
            #  ru: Это поле не может быть таким же как folder_name
            #  en: This field can't be the same as the folder_name
            raise InvalidDataSentError({'after': _('This field can\'t be the same as the target folder')})

        folder = get_object_or_404(Folder, user=self.request.user, name=folder_name, org=get_org())

        if folder.type != Folder.FOLDER_TYPE_CUSTOM:
            # Translators:
            #  ru: Эту папку нельзя перемещать
            #  en: This folder can't be moved
            raise InvalidDataSentError(_('This folder can\'t be moved'))

        after_folder = get_object_or_404(Folder, user=self.request.user, name=after_folder_name, org=get_org())

        if after_folder.type == Folder.FOLDER_TYPE_AUTO:
            # Translators:
            #  ru: Эту папку нельзя перемещать после указанного столбца
            #  en: The folder can't be moved after required folder
            raise InvalidDataSentError(_('The folder can\'t be moved after required folder'))

        qs = Folder.objects.filter(user=self.request.user, index_number__gt=after_folder.index_number, org=get_org())

        if after_folder.index_number < folder.index_number:
            qs.filter(index_number__lt=folder.index_number)

        qs.update(index_number=F('index_number') + 1)
        folder.index_number = after_folder.index_number + 1
        folder.save()

        return ok_response()
