import logging
import mimetypes
from enum import Enum

from django.conf import settings
from django.db import transaction
from django.http import Http404, HttpResponseForbidden, HttpResponseNotFound, HttpResponseRedirect
from django.utils.translation import ugettext_lazy as _
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.errors.rest_api_error import BillingLimitExceeded, ResourceIsMissing
from wiki.api_core.framework import PageAPIView, Redirect
from wiki.api_core.logic import files as upload_logic
from wiki.api_core.raises import raises
from wiki.api_core.utils import paginated
from wiki.api_frontend.serializers.files import AttachFileSerializer, FilesArraySerializer
from wiki.api_frontend.serializers.io import EmptySerializer, ok_response
from wiki.arc_compat import read_asset, trueish
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.files import logic as files_logic
from wiki.files.api import file_download_response
from wiki.files.models import File
from wiki.pages.access import has_access, is_admin
from wiki.pages.models import Page
from wiki.utils.supertag import remove_trailing_dots, translit

logger = logging.getLogger(__name__)


class PlaceholderImage(Enum):
    IMAGE_NOT_FOUND = 0
    IMAGE_ACCESS_DENIED = 1


img_cache = {}


def get_image(path: str):
    if path not in img_cache:
        try:
            img_cache[path] = read_asset(path)
        except FileNotFoundError:
            return b''

    return img_cache[path]


def guess_mimetype(file_url):
    """
    Вернуть mimetype документа.

    :param file_url:
    :rtype: str
    """
    mimetype, _nothing = mimetypes.guess_type(file_url)
    return mimetype or ''


RESPONSES = {
    403: (HttpResponseForbidden, settings.IMAGE_ACCESS_DENIED),
    404: (HttpResponseNotFound, settings.IMAGE_NOT_FOUND),
}


def prepare_response(code: int, use_image: bool):
    Klass, img_path = RESPONSES[code]
    if use_image:
        return Klass(get_image(img_path), content_type=guess_mimetype(img_path))
    return Klass()


class FilesListView(PageAPIView):
    """
    View для работы со списком файлов, прикрепленных к странице.
    """

    @raises()
    @paginated
    def get(self, request, start=None, limit=None, files_for_view=None, *args, **kwargs):
        """
        Вернуть список файлов, прикрепленных к странице.

        Принимает GET-параметры start и limit для пагинации.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/hanna/banana/.files?start=10&limit=20"
        %%

        Пример ответа:

        %%(js)
        {
            "debug": {
                ....
            },
            "data": {
                "total": 1,
                "data": [
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "описалово",
                        "url": "/wiki/ProAttachi/.files/rabotasgit.pdf",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "0.13",
                        "user_name": "user3371",
                        "name": "Работа с Git.pdf",
                        "can_delete": True,
                    }
                ]
            },
            "user": {
                ....                                    // пользователь, сделавший запрос
            }
        }
        %%
        """
        return Response(
            FilesArraySerializer(
                self.request.page,
                context={'user': self.request.user},
                start=start,
                limit=limit,
                files_for_view=files_for_view,
            ).data
        )


class FileNotFound(Http404):
    """
    Http 404
    """


class FileView(PageAPIView):
    """
    View для скачивания файла (точнее, для получения ссылки в downloader).
    """

    ERROR_404 = settings.IMAGE_NOT_FOUND
    ERROR_403 = settings.IMAGE_ACCESS_DENIED

    check_readonly_mode = True

    # чтобы в наследниках не приходилось импортировать file_for_page_by_filename.
    @staticmethod
    def get_file_by_filename(filename, page):
        return files_logic.file_for_page_by_filename(filename, page)

    def check_page_exists(self):
        if hasattr(self.request, 'redirect_to'):
            # если тэг в урле невалидный и надо средиректить на исправленный тэг, то здесь мы не можем просто подставить
            # исправленное значение тега, так как после тэга еще есть '/.files/имя.файла', поэтому берем из запроса весь
            # путь, исправляем и подставляем его.
            raise Redirect(redirect_to='/' + remove_trailing_dots(self.request.META['PATH_INFO']))
        super(FileView, self).check_page_exists()

    def check_page_access(self):
        """
        Здесь кастомная проверка, которая учитывает редиректы
        """
        return

    def handle_exception(self, exc):
        """
        Нам надо обработать 404 ответы и 403 по-особому, их получает прямо браузер.
        """
        if self.request.method == 'GET':
            if isinstance(exc, FileNotFound):
                return prepare_response(404, use_image=File.is_image(exc.args[1]))
            elif isinstance(exc, Http404):
                # вернуть пустой ответ без картинки.
                return prepare_response(404, use_image=False)
            elif isinstance(exc, UserHasNoAccess):
                # всегда вернуть картинку. Потому что легче всегда вернуть картинку 403.
                return prepare_response(403, use_image=True)
            elif isinstance(exc, Redirect):
                return HttpResponseRedirect(exc.redirect_to)
        return super(FileView, self).handle_exception(exc)

    @raises()
    def get(self, request, filename, *args, **kwargs):
        """
        Возвращает файл.

        Может вернуть 403, если у пользователя нет прав
        на страницу, или 404, если права есть, но файл не найден.

        Если запрошенный файл похож на картинку, логика обработки 404, 403 другая.
        Вернет картинку с числом 404, если картинки нет, вернет картинку с числом
        403, если нет прав доступа на чтение картинки.

        Если запрошенный файл не похож расширением на картинку, а файла нет,
        то пользователь получит пустой 404 ответ. Если же нет прав на просмотр файла,
        то пользователь в ответ получит все равно картинку с числом 403 и статус-код
        ответа будет 403.

        %%
        GET /_api/frontend/wat/.files/killa.gorilla
        %%
        """
        force_download = str(request.GET.get('download', '0')) == '1'
        file = self.get_file_by_filename(filename, request.page)

        if not file:
            logger.debug('No such file "%s" at page "%s"', filename, request.page.supertag)
            raise FileNotFound('No such file', translit(filename))

        if not file.page.has_access(request.user):
            raise UserHasNoAccess

        if file.storage_url is None:
            return HttpResponseNotFound()  # битый недо-загруженный файл.

        return file_download_response(file, force_download)

    @raises()
    def delete(self, request, filename, *args, **kwargs):
        """
        Удалить файл.

        Может вернуть 403, если у пользователя нет прав на страницу, 404, если права есть,
        но файл не найден, или 403, если файл найден, но автор
        страницы или файла — другой пользователь.

        %%
        DELETE /_api/frontend/wat/.files/killa.gorilla
        %%
        """
        file = self.get_file_by_filename(filename, request.page)
        if file is None:
            logger.debug('No such file "%s" at page "%s"', filename, request.page.supertag)
            raise Http404('No such file')

        if (
            request.user not in request.page.get_authors()
            and request.user.id != file.user.id
            and not is_admin(self.request.user)
        ):
            logger.info(
                'User %s has no right to delete file "%s" at page "%s"',
                request.user.username,
                file.url,
                request.page.supertag,
            )
            raise UserHasNoAccess(
                # Translators:
                #  ru: Пользователь не автор страницы и не владелец файла
                #  en: User is neither the page author nor the file owner
                _('User is neither the page owner nor the file owner')
            )

        files_logic.delete(file, user=request.user, page=request.page)

        # Cloudsearch part:
        CLOUD_SEARCH_CLIENT.on_model_delete(file)

        return ok_response()


class FileUploadView(PageAPIView):
    """
    View для загрузки файлов в storage
    """

    available_content_types = ('multipart/form-data', 'application/octet-stream')
    serializer_class = EmptySerializer

    def check_page_exists(self):
        # загрузка файлов происходит без указания страницы
        pass

    def check_page_access(self):
        # загрузка файлов происходит без указания страницы
        pass

    @raises(BillingLimitExceeded)
    def post(self, *args, **kwargs):
        """
        POST-handler, загружает в хранилище первый файл из запроса.
        Файлы в запросе должны быть такие же, как посылает html форма -
        запрос с content-type: multipart/form-data;

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: multipart/form-data" \
        "https://wiki-api.yandex-team.ru/_api/frontend/.upload" -F file=@111.txt
        %%

        Возвращает json c storage_id файла вида
        %%(json)
        {
            "data": {
                "storage_id": "<file storage_id string>"
            }
        }
        %%
        """
        try:
            storage_id = upload_logic.upload_file_to_storage(self.request)
        except upload_logic.FileUploadError as exc:
            raise InvalidDataSentError(str(exc))
        return Response({'storage_id': storage_id})


class AttachFileView(FilesListView):
    """
    View для прикрепления к странице файлов, загруженных в хранилище
    через ручку %%/.upload%%.
    """

    serializer_class = AttachFileSerializer
    check_readonly_mode = True

    def check_page_access(self):
        """
        Проверить, что у пользователя есть права на запись.
        """
        if self.request.from_yandex_server:
            raise UserHasNoAccess
        if not (
            is_admin(self.request.user)
            or has_access(
                self.request.supertag,
                self.request.user,
                privilege='write',
                current_page=self.request.page,
            )
        ):
            raise UserHasNoAccess

    @transaction.atomic
    @raises()
    def post(self, request, *args, **kwargs):
        """
        Добавить на страницу файлы, загруженные ранее в хранилище.
        Вернуть список файлов, прикрепленных к странице текущим запросом.
        Формат ответа полностью совпадает с форматом ручки %%/.files%%.

        Пример запроса:

        %%(http)
        POST /_api/frontend/users/elisei/test/.attach
        %%

        Тело запроса:

        %%(js)
        {
            // список id файлов (storage_id, полученные из ручки /.upload)
            "files": ["wiki:file:1231:126845:2014-04-09 20:51:46:593471", "..."]
        }
        %%

        Пример запроса из командной строки:

        %%(sh)
        curl -v -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json"
        "https://wiki-api.yandex-team.ru/_api/frontend/page1/test/.attach"
        --data '{"files": ["108159/wiki:file:MTExLnR4dA==:623:2018-09-05 12:57:20:048714:2478070"]}'
        %%

        Пример ответа:

        %%(js)
        {
            "debug": {
                ....
            },
            "data": {
                "total": 1,
                "data": [
                    {
                        "upload_date": "2014-04-10 12:10:33",
                        "description": "описалово",
                        "url": "/wiki/.files/rabotasgit.pdf",
                        "docviewer_url": "http://docviewer.dst.yandex-team.ru/?url=",
                        "size": "0.13",
                        "user_name": "user3371",
                        "name": "Работа с Git.pdf"
                    }
                ]
            },
            "user": {
                ....                                    // пользователь, сделавший запрос
            }
        }
        %%
        """
        page = Page.objects.select_for_update().get(pk=self.request.page.pk)
        data = self.validate()

        files = data['files']

        is_silent = request.GET.get('is_silent') in trueish

        # проверить права на запись
        if not self.request.page.has_access(self.request.user, privilege='write'):
            raise UserHasNoAccess

        for storage_id in files:
            self._save_file(page, storage_id, is_silent=is_silent)

        kwargs['files_for_view'] = files

        return self.get(request, *args, **kwargs)

    def _save_file(self, page, storage_id, is_silent: bool = False):
        """
        Сохранить информацию о прикрепленном файле в базе.
        """
        try:
            file = upload_logic.add_file_to_page(page, self.request.user, storage_id, is_silent)

            # Cloudsearch part:
            CLOUD_SEARCH_CLIENT.on_model_upsert(file)
        except upload_logic.NoFileInStorage as e:
            raise ResourceIsMissing(str(e))
        except Exception as e:
            logger.exception(e)
