from django.db.models import Q, QuerySet

from rest_framework.pagination import PageNumberPagination
from rest_framework.response import Response
from rest_framework.serializers import CharField, ModelSerializer

from wiki.api_core.errors.rest_api_error import ResourceIsMissing
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.org import get_org
from wiki.pages.access import get_bulk_access_status, ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED
from wiki.pages.models import Page

from typing import Any, Generator, List


class Pagination(PageNumberPagination):
    page_size = 100
    page_size_query_param = 'page_size'
    max_page_size = 500


class PageSerializer(ModelSerializer):
    supertag = CharField()

    class Meta:
        model = Page
        fields = ('id', 'supertag')


def create_queryset(supertag: str) -> QuerySet:
    org = get_org()
    return Page.active.filter(Q(supertag=supertag, org=org) | Q(supertag__startswith=supertag + '/', org=org))


def filter_by_access(pages: List[Page], user) -> Generator[Page, Any, None]:
    access_statuses = get_bulk_access_status(tags=[p.supertag for p in pages], user=user)
    valid_access = {ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED}
    return (page for page in pages if access_statuses[page.supertag] in valid_access)


class FlatListPages(WikiAPIView):
    pagination_class = Pagination

    @raises()
    def get(self, request, tag: str, **kwargs) -> Response:
        """
        Получить плоский список вложенных страниц для заданной страницы c паджинацией и проверкой прав доступа

        %%(sh)
        curl -H "Authorization: OAuth <token>"
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.children?page=<page_num>&page_size=<page_size>"
        %%

        Ответ
        %%(js)
        {
            "results": [
                {
                    "id": 1,
                    "supertag": "root"
                },
                {
                    "id": 2,
                    "supertag": "root/a"
                },
            ]
        }
        %%
        """

        if request.page is None:
            raise ResourceIsMissing(f'Incorrect "{tag}" tag not found')

        pages_queryset = create_queryset(supertag=request.page.supertag)
        pages_paginated = self.paginate_queryset(pages_queryset)
        pages_accessible = filter_by_access(pages_paginated, user=request.user)

        return Response({'results': PageSerializer(pages_accessible, many=True).data})
