import logging

from django.conf import settings
from django.http import HttpResponseBadRequest
from rest_framework.response import Response

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.logic.forced_sync import forced_sync
from wiki.api_frontend.serializers.forced_sync import ForcedSyncRequest

logger = logging.getLogger(__name__)


class ForcedSyncView(WikiAPIView):
    """
    View для запуска и контроля выполнения процедуры форсированной синхронизации данных пользователя
    и его организации.
    """

    serializer_class = ForcedSyncRequest

    available_content_types = ['application/json']

    @raises()
    def post(self, request, *args, **kwargs):
        """
        Послать запрос для запуска или проверки статуса процедуры форсированной синхронизации данных пользователя
        и его организации.
        Параметрах POST запроса: org_id или user_iam_token или (cloud_org_id и user_iam_token), cloud_uid или uid.

        Пример запроса:

        %%(sh)
        curl -H "Content-Type: application/json" -X "POST" -H "X-Ya-Service-Ticket: ..."
        "https://wiki-api.yandex.ru/_api/frontend/.forced_sync"
        --data '{"org_id": "111", "user_iam_token": "", "cloud_uid": "aje009akrl98vg3rf1hb", "uid": "9000000000000113"}'
        %%

        Тело запроса:

        %%(js)
        {
            "org_id": "111",
            "user_iam_token": "",
            "cloud_uid": "aje009akrl98vg3rf1hb",
            "uid": "9000000000000113"
        }
        %%

        Ответ:

        %%(js)
        {
            "data": {
                "result": "ok"|"failed"|"in_progress"
            }
        }
        %%
        """
        if not settings.IS_BUSINESS:
            return HttpResponseBadRequest(content='This method for business environment only')

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        org_id = serializer.validated_data.get('org_id', '')
        cloud_org_id = serializer.validated_data.get('cloud_org_id', '')
        user_iam_token = serializer.validated_data.get('user_iam_token', '')
        cloud_uid = serializer.validated_data.get('cloud_uid', '')
        uid = serializer.validated_data.get('uid', '')
        force = serializer.validated_data.get('force', False)
        result = forced_sync(org_id, cloud_org_id, cloud_uid, uid, user_iam_token, skip_check=force)
        logger.info(f'Forced sync response = {result.dict()}')
        return Response(data=result.dict(), status=200)
