import logging
import urllib.error
import urllib.parse
import urllib.request

from django.db import OperationalError
from django.http import Http404, HttpResponse
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _
from django_pgaas import atomic_retry
from psycopg2 import errorcodes as pg_errorcodes
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.logic import GridProxy
from wiki.api_frontend.serializers.grids import (
    GridCloneSerializer,
    GridConcurrentChangesSerializer,
    GridDataSerializer,
    GridRevisionSerializer,
    ModifyingSequenceSerializer,
    errors,
)
from wiki.api_frontend.serializers.io import EmptySerializer
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.grids.consts import GridContentSchema
from wiki.grids.logic import changes, operations
from wiki.grids.models import Grid, Revision
from wiki.grids.utils.export import export_csv, export_docx, export_xls
from wiki.pages import access
from wiki.pages.logic import hierarchy
from wiki.pages.models import Page
from wiki.utils.supertag import tag_to_supertag

logger = logging.getLogger(__name__)


class GridDataView(PageAPIView):
    """
    View для получения данных грида (без метаинформации о странице)


    """

    expected_page_type = Page.TYPES.GRID

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Получить всю структуру табличного списка.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.grid"
        %%

        Пример ответа:
        %%(json)
        {
          "structure": {
            "sorting": [
              {
                type: "asc",
                name: "2"
              }
            ],
            "title": "grid",
            "fields": [
              {
                "title": "Камент",
                "required": false,
                "type": "string",
                "sorting": "asc",
                "name": "2"
              },
            "done": false,
          },
          "rows": [
            [
              {
                "sort": "123 456",
                "raw": "123 456 ",
                "b64wom1": "...",
                "transformed": "...",
                "row_id": "7",
                "__key__": "2"
              }
            ]
          ],
          "version": "6005994"
        }
        %%
        """
        page = self.request.page
        page = Grid.objects.get(pk=page.pk)
        return Response(get_grid_data(page, request))


def get_grid_data(grid, request):
    grid_proxy = GridProxy(grid, user_auth=request.user_auth, get_params=request.GET)
    selected_fields = request.GET.get('_fields')
    if selected_fields:
        selected_fields = selected_fields.split(',')
    serializer = GridDataSerializer(grid_proxy, only=selected_fields)

    return serializer.data


def get_grid_data_api_v2(grid, request, revision):
    if revision:
        grid.body = revision.body
    grid_proxy = GridProxy(grid, user_auth=request.user_auth, get_params=request.GET)
    return GridContentSchema.from_orm(grid_proxy)


class GridRowsView(PageAPIView):
    """
    View для работы со строками гридов
    """

    serializer_class = ModifyingSequenceSerializer
    expected_page_type = Page.TYPES.GRID
    check_readonly_mode = True

    @atomic_retry()
    @raises(
        errors.MaximumRowsCountExceeded,
        errors.ObjectIsLockedForUpdate,
    )
    def post(self, request, *args, **kwargs):
        """
        Добавить строку к гриду, в начало либо после определенной строки.

        Пример запроса, **добавляющего строку** перед всеми (в начало):

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/hanna/banana/.grid/rows" \
        --data 'тело запроса'
        %%

        Пусть в табличном списке из примера есть поля latest_film, number, year_of_birth,
        cool_actor. Тогда для сохранения надо послать такой запрос:

        тело запроса
        %%(js)
        {
          "version": "43",
          "changes": [{  // тут может быть несколько подряд идущих изменений.
            "added_row": {
              "after_id": "-1",
              "data": {
                // перечислить набор полей, которые надо заполнить
                // если ничего заполнять не надо - передайте data: {}
                "latest_film": "Boston Legal",  // тип поля - строка
                "number": 10,                   // тип поля число
                "year_of_birth": ["1950"],      // тип поля список
                "cool_actor": true,             // тип поля список
              }
            }
          }]
        }
        %%

        Пример ответа:
        %%(js)
        {
          "version": "44",
          "success": true,
          "id": '1'
        }
        %%

        Если в changes передать несколько изменений, то они либо будут применены все,
        либо ни одно из них.

        Используется фронтэндом для добавления пустых строк к гриду. Он всегда передает пустой
        маппинг ##"{}"##.
        """
        try:
            grid = Grid.objects.select_for_update().get(id=request.page.id)
        except Grid.DoesNotExist:
            raise Http404
        except OperationalError as err:
            if err.__cause__.pgcode == pg_errorcodes.LOCK_NOT_AVAILABLE:
                # не пятисотить, если не можем взять select_for_update у грида в течение lock_timeout и после ретраев.
                # WIKI-12087: фикс 500-к и ошибки в логах "OperationalError: canceling statement due to lock timeout"
                raise errors.ObjectIsLockedForUpdate()
            else:
                raise

        serializer = self.get_serializer(data=request.data)

        if serializer.is_valid():
            serializer.save(author_of_changes=self.request.user, grid=grid)
            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(grid)
            return Response(
                {'version': grid.get_page_version(), 'success': True, 'id': str(grid.access_meta['autoincrement'])}
            )
        else:
            logger.warning('Invalid request sent to grids "%d": "%s"', grid.id, serializer.errors)
            raise InvalidDataSentError(serializer.errors)


class GridExportView(PageAPIView):
    """
    View для экспорта гридов.
    """

    expected_page_type = Page.TYPES.GRID

    @staticmethod
    def format_content_disposition_header(filename):
        return '''attachment; filename="{0}"; filename*="UTF-8''{1}"'''.format(
            filename,
            urllib.parse.quote(filename),
        )

    @raises()
    def get(self, request, format_type, *args, **kwargs):
        """
        Отдать пользователю табличный список, преобразованный в файл заданного формата.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.grid/export/<format>"
        %%

        format может быть "csv", "xls" или "docx".
        """
        page = self.request.page

        # табличный список
        grid = Grid.objects.get(pk=page.pk)

        data = grid.get_rows(self.request.user_auth)
        structure = grid.access_structure

        filename = grid.tag.split('/')[-1]

        if format_type == 'csv':
            filename += '.csv'
            mimetype = 'text/csv'
            method = export_csv
        elif format_type == 'xls':
            filename += '.xls'
            mimetype = 'application/vnd.ms-excel'
            method = export_xls
        else:
            filename += '.docx'
            mimetype = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
            method = export_docx

        file_contents = method(structure, data)

        response = HttpResponse(file_contents.getvalue(), content_type=mimetype)
        response['Content-Disposition'] = self.format_content_disposition_header(filename).encode()
        return response


class GridRevisionView(PageAPIView):
    """
    View для получения данных грида в конкретной ревизии (без метаинформации о странице)
    """

    serializer_class = GridRevisionSerializer
    expected_page_type = Page.TYPES.GRID

    @raises()
    def get(self, request, revision_id, *args, **kwargs):
        """
        Получить ревизию табличного списка по заданному ID (первичному ключу) ревизии.

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/hanna/banana/.grid/revisions/42"
        %%

        Пример ответа (те поля, которые не описаны смотри в ручке получения табличного списка):
        %%(json)
        {
          "structure": ...,
          "rows": ...,
          "version": "6005994",
          "author": {...},
          "created_at": "2014-03-13T13:39:53"
        }
        %%
        """
        revision = get_object_or_404(Revision, id=revision_id, page=request.page)
        return Response(self.serializer_class(GridProxy(revision, user_auth=self.request.user_auth)).data)


class GridConcurrentChangesView(PageAPIView):
    """
    View для получения обзорной информации об изменениях грида, случившихся после указанной ревизии.
    """

    serializer_class = GridConcurrentChangesSerializer
    expected_page_type = Page.TYPES.GRID

    @raises()
    def get(self, request, version, *args, **kwargs):
        """
        Получить обзорную информацию об изменениях грида, случившихся после указанной версии.

        Пример запроса, получения обзора изменений после версии 42:
        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/mygrid/.changes/42"
        %%

        Пример ответа:
        %%(js)
        {
          "editors": [     // список пользователей, менявших грид (кроме текущего пользователя)
            "login": "user1",
            "uid": 12345,
            "first_name": "Vasya",
            "name_name": "Pupkin",
          ]
        }
        %%
        """
        data = changes.get_grid_concurrent_changes(request.page, version, request.user)
        return Response(self.serializer_class(data).data)


class GridRollbackView(PageAPIView):
    """
    Откатить грид на старую версию.

    Пример ответа: см. GridRevisionView

    """

    serializer_class = EmptySerializer
    response_serializer_class = GridRevisionSerializer
    expected_page_type = Page.TYPES.GRID
    # не надо проверять заголовок content-type
    # потому что в теле POST ничего не передается.
    check_content_type_for = []
    check_readonly_mode = True

    @raises()
    def post(self, request, revision_id, *args, **kwargs):
        """
        Откатить грид на старую версию.

        %%(sh)
        curl -X POST -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/hanna/banana/.grid/revisions/42/rollback"
        %%

        Пример ответа (те поля, которые не описаны смотри в ручке получения табличного списка):
        %%(json)
        {
          "structure": ...,
          "rows": ...,
          "version": "6005994",
          "author": {...},
          "created_at": "2014-03-13T13:39:53"
        }
        """
        from wiki.grids.utils.revisions import rollback_grid

        page = request.page
        revision = get_object_or_404(Revision, id=revision_id, page=page)

        rollback_grid(
            grid=page,
            revision=revision,
            user=request.user,
        )

        # TODO: WIKI-9685 - возможно, перед отдачей ревизии ее нужно переформатировать свежим WF

        # Cloudsearch
        CLOUD_SEARCH_CLIENT.on_model_upsert(page)

        return Response(self.response_serializer_class(GridProxy(revision, user_auth=self.request.user_auth)).data)


class GridCloneView(PageAPIView):
    """
    Клонировать грид
    """

    serializer_class = GridCloneSerializer

    expected_page_type = Page.TYPES.GRID
    check_content_type_for = []
    check_readonly_mode = True

    @raises(UserHasNoAccess)
    def post(self, request, *args, **kwargs):
        """
        Клонировать грид с контентом или без

        %%(sh)
        curl -X POST -H "Authorization: OAuth <token>" \
        "https://wiki-api.yandex-team.ru/_api/frontend/banana/.grid/clone" \
        --data '{"destination": "hanna", "with_data": true}'
        %%

        ответ:
        %%(js)
        {
            "success": True,
            "message": "",
        }
        %%
        """
        grid = Grid.objects.get(id=request.page.id)
        data = self.validate()

        destination_tag = data['destination']
        destination_supertag = tag_to_supertag(destination_tag)
        nearest_parent = hierarchy.get_nearest_existing_parent(destination_supertag)

        user_has_access = (
            not nearest_parent or nearest_parent.has_access(self.request.user) or access.is_admin(self.request.user)
        )
        if not user_has_access:
            # Translators:
            #  ru: Запрещено, потому что у вас не будет доступа к табличному списку
            #  en: Forbidden, cause you will not have access to the grid
            raise UserHasNoAccess(_('Forbidden, cause you will not have access to the grid'))

        operations.clone(
            grid=grid,
            user=request.user,
            destination_tag=destination_tag,
            with_data=data['with_data'],
        )

        # Cloudsearch
        CLOUD_SEARCH_CLIENT.on_model_upsert(grid)

        return self.build_success_status_response()
