
import logging

from django.db import transaction
from rest_framework.response import Response

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView, WikiAPIView
from wiki.api_core.logic import files as upload_logic
from wiki.api_core.raises import raises
from wiki.api_frontend.logic.errors import simplify_grid_row_errors
from wiki.api_frontend.serializers.grids import ImportDataParsePreviewSerializer, ImportDataPreviewSerializer, errors
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.grids.logic import grids_import
from wiki.pages.models import Page

logger = logging.getLogger(__name__)


def raise_invalid_data_sent_error(grid_import_error):
    """
    @type grid_import_error: GridImportError
    """
    non_field_messages = simplify_grid_row_errors(grid_import_error.errors or {})
    if grid_import_error.detail:
        non_field_messages.insert(0, grid_import_error.detail)
    raise InvalidDataSentError(non_field_messages=non_field_messages)


class ImportFileUploadView(WikiAPIView):
    """
    Импорт данных в табличный список.

    View для загрузки импортируемого файла в кэш.
    """

    available_content_types = ('multipart/form-data', 'application/octet-stream', 'application/x-www-form-urlencoded')

    @raises(
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
    )
    def post(self, *args, **kwargs):
        """
        Загрузить в кэш первый файл из запроса.
        Файл в запросе должен быть таким же, как посылает html форма - запрос с content-type: multipart/form-data;
        Поддерживаемые расширения файлов: xls, xlsx, csv, txt.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%charset%% | str | опциональный | кодировка данных. По умолчанию: автоопределение ||
        || %%delimiter%% | str | опциональный | разделитель полей. По умолчанию: ';' ||
        || %%quotechar%% | str | опциональный | ограничитель строк. По умолчанию: '"' ||
        || %%omit_first%% | boolean | опциональный | признак необходимости импорта первой строки с заголовками.
        По умолчанию: false ||
        |#
        Параметры могут передаваться как в строке запроса, так и в теле запроса.

        Возвращает: данные для предварительного просмотра результата разбора данных из файла (первые 10 строк).

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: multipart/form-data" \
        --form file1=@filename press=submit \
        "https://wiki-api.yandex-team.ru/_api/frontend/.import/grid/upload?charset=utf-8&delimiter=%3B&quotechar=%22"
        %%

        Пример ответа:

        %%(json)
        {
            "debug": {
                ....                            // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "row_count": 2,                 // число строк в импортируемом файле
                "column_count": 3               // число столбцов в импортируемом файле
                "cache_key": 'fa5921',          // ключ в кэше с данными
                "rows": [                       // несколько строк из импортируемого файла в качестве примера
                    [
                        "column1",
                        "column2",
                        "column3"
                    ],
                    [
                        "текст ячейки 1",
                        "текст ячейки 2",
                        "123"
                    ]
                ],
            },
            "user": {
                ....
            }
        }
        %%

        Если пользователь попытается загрузить файл с неподдерживаемым расширением, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "'doc' is unsupported file extension. Valid values: ['xls', 'xlsx', 'csv', 'txt']",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если формат содержимого файла не является текстовым, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Invalid format of the file contents",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если пользователь не указал явно кодировку содержимого файла, а сервер не смог ее определить автоматически,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Can't determine import file encoding. Please specify the encoding explicitly",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если при перекодировании импортируемых данных произошла ошибка, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Can't encode import file content. Please try to specify another encoding",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        try:
            cache_key, name, size = upload_logic.upload_file_to_cache(self.request)
        except upload_logic.FileUploadError as exc:
            raise InvalidDataSentError(str(exc))
        except grids_import.InvalidImportDataError as exc:
            raise InvalidDataSentError(errors_or_message=exc.errors, non_field_messages=exc.detail)

        try:
            grids_import.validate_import_data(self.request, cache_key)
            preview_data = grids_import.get_parse_preview_data(self.request, cache_key)
        except grids_import.InvalidImportDataError as exc:
            raise InvalidDataSentError(errors_or_message=exc.errors, non_field_messages=exc.detail)

        return Response(ImportDataParsePreviewSerializer(preview_data).data)


def get_cache_key_or_raise(request):
    """
    Достать ключ либо из QUERY_STRING либо из тела запроса.
    """
    cache_key = request.data.get('key') or request.GET.get('key')
    if not cache_key:
        raise InvalidDataSentError('key is a mandatory parameter')
    return cache_key


class ImportFileDataParseView(WikiAPIView):
    """
    Импорт данных в табличный список.

    View для разбора импортируемых данных, находящихся в кэше.
    """

    @raises(
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
    )
    def get(self, request, *args, **kwargs):
        """
        Распарсить импортируемые данные с применением переданных настроек или настроек по умолчанию.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%key%% | str | !!обязательный!! | ключ в кэше, полученный в ответе при загрузке файла ||
        || %%charset%% | str | опциональный | кодировка данных. По умолчанию: автоопределение ||
        || %%delimiter%% | str | опциональный | разделитель полей. По умолчанию: ';' ||
        || %%quotechar%% | str | опциональный | ограничитель строк. По умолчанию: '"' ||
        || %%omit_first%% | boolean | опциональный | признак необходимости импорта первой строки с заголовками.
        По умолчанию: false ||
        |#

        Возвращает: данные для предварительного просмотра результата разбора данных из файла (первые 10 строк).

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/.import/grid/parse?key=69edc5&charset=utf-8&delimiter=%3B&quotechar=%22"
        %%

        Пример ответа:

        %%(json)
        {
            "debug": {
                ....                            // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "row_count": 2,                 // число строк в импортируемом файле
                "column_count": 3               // число столбцов в импортируемом файле
                "cache_key": 'fa5921',          // ключ в кэше с данными
                "rows": [                       // несколько строк из импортируемого файла в качестве примера
                    [
                        "column1",
                        "column2",
                        "column3"
                    ],
                    [
                        "текст ячейки 1",
                        "текст ячейки 2",
                        "123"
                    ]
                ],
            },
            "user": {
                ....
            }
        }
        %%

        Если пользователь попытается загрузить файл с неподдерживаемым расширением, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "'doc' is unsupported file extension. Valid values: ['xls', 'xlsx', 'csv', 'txt']",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если формат содержимого файла не является текстовым, то в ответе будет передана
        ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Invalid format of the file contents",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если пользователь не указал явно кодировку содержимого файла, а сервер не смог ее определить автоматически,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Can't determine import file encoding. Please specify the encoding explicitly",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если при перекодировании импортируемых данных произошла ошибка, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": "Can't encode import file content. Please try to specify another encoding",
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если в кэше по переданному ключу key не найдено содержимое, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'key': 'No content in cache for the specified key or the key is already expired'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        cache_key = get_cache_key_or_raise(request)

        try:
            grids_import.validate_import_data(self.request, cache_key)
            parse_preview_data = grids_import.get_parse_preview_data(self.request, cache_key)
        except grids_import.InvalidImportDataError as err:
            raise_invalid_data_sent_error(err)

        return Response(ImportDataParsePreviewSerializer(parse_preview_data).data)


class ImportDataView(PageAPIView):
    """
    Импорт данных в табличный список.

    View для настройки параметров импорта, предварительного просмотра и сохранения результата.
    """

    def check_page_exists(self):
        """
        Может вызываться для несуществующей страницы
        """
        self.expected_page_type = Page.TYPES.GRID if self.request.page else None

        return True

    @raises(
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
    )
    def get(self, request, *args, **kwargs):
        """
        Импортировать данные в табличный список с учетом переданных настроек и вернуть данные для
        предварительного просмотра результатов импорта.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%key%% | str | !!обязательный!! | ключ в кэше, полученный в ответе при загрузке файла ||
        || %%charset%% | str | опциональный | кодировка данных. По умолчанию: автоопределение ||
        || %%delimiter%% | str | опциональный | разделитель полей. По умолчанию: ';' ||
        || %%quotechar%% | str | опциональный | ограничитель строки. По умолчанию: '"' ||
        || %%omit_first%% | boolean | опциональный | признак необходимости импорта первой строки с заголовками.
        По умолчанию: false ||
        || %%icolumn_0_to%% | str | опциональный | имя столбца в существующем табличном списке, в который надо
        импортировать столбец. Цифра в имени параметра - индекс столбца из списка import_columns.
        Если данные импортируются в новый столбец, то параметр не передается. По умолчанию: отсутствует ||
        || %%icolumn_0_type%% | str | опциональный | тип данных для нового столбца, куда будет импортироваться выбранный
        столбец. Цифра в имени параметра - индекс столбца из списка import_columns.
        Если данные импортируются в уже существующий столбец, то параметр не передается. По умолчанию: 'string' ||
        || %%icolumn_0_enabled%% | boolean | опциональный | признак необходимости импорта столбца.
        Цифра в имени параметра - индекс столбца из списка import_columns. По умолчанию: true ||
        |#

        Возвращает: данные для предварительного просмотра результата импорта данных в табличный список
        (первые 3 строки).

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/<tag>/.grid/import?key=69edc5&charset=utf-8&delimiter=%3B&quotechar=%22"
        %%

        Пример ответа:

        %%(json)
        {
            "debug": {
                ....                                        // дебаг информация, в продакшне может быть отключена
            },
            "data": {
                "row_count": 1,
                "cache_key": "89330a",
                "column_count": 2
                "import_columns": [                         // настройки для импорта столбцов
                    {
                        "icolumn_0_enabled": True,
                        "icolumn_0_to": "name",
                        "title": "Name of conference",
                        "column_name": "icolumn_0",
                        "icolumn_0_type": "string"
                    },
                    {
                        "icolumn_1_type": "string",
                        "column_name": "icolumn_1",
                        "icolumn_1_enabled": True,
                        "icolumn_1_to": None,
                        "title": "column 2"
                    }
                ],
                "column_data_types": [   // список возможных типов данных для нового столбца
                    [
                        "string",                           // наименование типа данных
                        "Text"                              // значение для отображения
                    ],
                    ....
                ],
                "existing_columns": [    // список существующих столбцов для выбора, куда можно импортировать данные
                    [
                        "name",                             // имя столбца
                        "Name of conference"                // заголовок столбца
                    ],
                    ....
                ],
                "existing_rows": [      // пример строк с данными из существующей таблицы (первые 3 строки)
                    [
                        {
                            "sort": "row n1",
                            "type": "string",
                            "value": "{"content":[{"wiki-attrs":{"txt":"row N1","pos_start":0,"pos_end":6}, ..."
                        },
                        {
                            "sort": "",
                            "type": "string",
                            "value": ""
                        }
                    ],
                    [
                    ....
                    ]
                ],
                "imported_rows": [                          // пример строк с импортированными данными (первые 3 строки)
                    [
                        {
                            "sort": "текст в ячейке 1",
                            "type": "string",
                            "value": "{"content":[{"wiki-attrs":{"txt":"текст в ячейке 1","pos_start":0,"pos_end":6}"
                        },
                        {
                            "sort": "текст в ячейке 2",
                            "type": "string",
                            "value": "{"content":[{"wiki-attrs":{"txt":"текст в ячейке 2","pos_start":0,"pos_end":6}..."
                        }
                    ]
                ],
                "fields": [                                 // описание структуры импортируемых столбцов
                    {
                        "sorting": True,
                        "name": "name",
                        "title": "Name of conference",
                        "required": True,
                        "width": "200px",
                        "type": "string"
                    },
                    {
                        "required": False,
                        "sorting": True,
                        "type": "string",
                        "name": "811",
                        "title": "column 1"
                    }
                ],
            },
            "user": {
                ....
            }
        }
        %%

        Если тип данных импортируемого столбца не соответствует типу данных выбранного существующего или нового столбца,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA', а в словаре errors в качестве значения
        будет список с тайтлами ошибочных столбцов - ['столбец 1'] или ['столбец 1', 'столбец в гриде']
        (если в списке одно значение, то это ошибка при импорте данных в новый столбец, а если в списке два значения,
        то это ошибка при импорте данных из импортруемого столбца с тайтлом 'столбец 1' в существующий столбец
        с тайтлом 'столбец в гриде'):
        %%(js)
        {
            "error": {
                "message": 'The form has been filled in incorrectly',
                "errors": {
                    'date': ['столбец 1']
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если в кэше по переданному ключу key не найдено содержимое, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'key': 'No content in cache for the specified key or the key is already expired'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        cache_key = get_cache_key_or_raise(request)

        try:
            grids_import.validate_import_data(self.request, cache_key)
            import_data_preview = grids_import.get_import_preview_data(self.request, cache_key, row_count_for_preview=2)
        except grids_import.InvalidImportDataError as err:
            raise_invalid_data_sent_error(err)

        return Response(ImportDataPreviewSerializer(import_data_preview).data)

    @raises(
        errors.MaximumRowsCountExceeded,
        errors.MaximumColumnsCountExceeded,
        errors.MaximumCellSizeExceeded,
    )
    @transaction.atomic()
    def post(self, request, *args, **kwargs):
        """
        Импортировать данные в табличный список с учетом переданных настроек и сохранить его.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%key%% | str | !!обязательный!! | ключ в кэше, полученный в ответе при загрузке файла ||
        || %%charset%% | str | опциональный | кодировка данных. По умолчанию: автоопределение ||
        || %%delimiter%% | str | опциональный | разделитель полей. По умолчанию: ';' ||
        || %%quotechar%% | str | опциональный | ограничитель строки. По умолчанию: '"' ||
        || %%omit_first%% | boolean | опциональный | признак необходимости импорта первой строки с заголовками.
        По умолчанию: false ||
        || %%icolumn_0_to%% | str | опциональный | имя столбца в существующем табличном списке, в который надо
        импортировать столбец. Цифра в имени параметра - индекс столбца из списка import_columns.
        Если данные импортируются в новый столбец, то параметр не передается. По умолчанию: отсутствует ||
        || %%icolumn_0_type%% | str | опциональный | тип данных для нового столбца, куда будет импортироваться выбранный
        столбец. Цифра в имени параметра - индекс столбца из списка import_columns.
        Если данные импортируются в уже существующий столбец, то параметр не передается. По умолчанию: 'string' ||
        || %%icolumn_0_enabled%% | boolean | опциональный | признак необходимости импорта столбца.
        Цифра в имени параметра - индекс столбца из списка import_columns. По умолчанию: true ||
        |#

        Возвращает: статус-код ответа.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" -X "POST" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<tag>/.grid/import"
        --data 'тело запроса'
        %%

        тело запроса
        %%(js)
        {
            "key": "69edc5",
            "charset": "utf-8",
            "delimiter": ";",
            "quotechar": '"',
            "omit_first": True,
            "icolumn_0_to": "name",
            "icolumn_1_enabled": False,
        }
        %%

        Если тип данных импортируемого столбца не соответствует типу данных выбранного существующего столбца,
        то в ответе будет передана ошибка с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": 'The form has been filled in incorrectly',
                "errors": {
                    'date': 'Enter a valid date.'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%

        Если в кэше по переданному ключу key не найдено содержимое, то в ответе будет передана ошибка
        с кодом 'CLIENT_SENT_INVALID_DATA':
        %%(js)
        {
            "error": {
                "message": 'Client sent invalid data',
                "errors": {
                    'key': 'No content in cache for the specified key or the key is already expired'
                },
                "error_code": "CLIENT_SENT_INVALID_DATA"
            }
        }
        %%
        """
        cache_key = get_cache_key_or_raise(request)

        try:
            grids_import.validate_import_data(self.request, cache_key)
            grid = grids_import.save_import_data(self.request, cache_key)

            # Cloudsearch
            CLOUD_SEARCH_CLIENT.on_model_upsert(grid)
        except grids_import.InvalidImportDataError as err:
            raise_invalid_data_sent_error(err)

        return self.build_success_status_response()
