import logging

from django.conf import settings
from django.http import Http404
from rest_framework.response import Response

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_core.utils import paginated
from wiki.api_frontend.serializers.users import DetailedGroupSerializer
from wiki.org import org_group
from wiki.users import dao
from wiki.users.models import Group

logger = logging.getLogger(__name__)


class GroupView(WikiAPIView):
    """
    Вьюшка для работы с группой пользователей
    """

    serializer_class = DetailedGroupSerializer

    def check_page(self):
        """
        нам не важно, на какой странице вызван данный хэндлер
        """
        pass

    def check_page_access(self):
        """
        права тоже подойдут любые, кроме Яндекс.Сервера
        """

        if self.request.from_yandex_server:
            raise UserHasNoAccess

    @raises()
    @paginated
    def get(self, request, group_id, *args, **kwargs):
        """
        Вернуть информацию о группе.

        Параметры:
        #|
        || **имя** | **тип** | **обязательность** | **описание** ||
        || %%is_department%% | boolean | опциональный | признак того, что искомая сущность является департаментом.
            Применяется только в Вики для Бизнеса, чтобы разделить поиск групп и департаментов, так как в Директории -
            это разные сущности с пересекающимися идентификаторами.
        |#

        %%(sh)
        curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/<любой тег>/.groups/<group id>"
        %%
        Ответ:
        %%(json)
        {
          "data": {
            "id": 120,
            "name": "Группа",
            "url": "yandex_grouppa",  // только для инстансов Вики в Интранете
            "externals_count": 2,  // только для инстансов Вики в Интранете
            "type": "__service__",
            "title": "Отдел продаж",  // только для инстансов Вики для Бизнеса
            "dir_id": "3232",  // только для инстансов Вики для Бизнеса
            "users": [
                { пользователь в стандартном формате },
                { пользователь в стандартном формате },
                ...
            ]
          }
        }
        %%
        """
        try:
            filter_params = {dao.get_id_attr_name(): group_id}
            if settings.IS_BUSINESS:
                from wiki.users_biz.models import GROUP_TYPES

                is_department = request.GET.get('is_department', 'False')[0].upper() == 'T'
                filter_params['group_type'] = GROUP_TYPES.department if is_department else GROUP_TYPES.group
            group = org_group().get(**filter_params)
        except Group.DoesNotExist:
            raise Http404
        return Response(self.serializer_class(group).data)
