from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_core.framework import PageAPIView
from wiki.api_core.raises import raises
from wiki.api_frontend.serializers.keywords import KeywordsSerializer
from wiki.pages.logic import keywords as keywords_logic


class KeywordsView(PageAPIView):
    serializer_class = KeywordsSerializer
    available_for_admins = True

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Список ключевых слов для страницы.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
             -H "Content-Type: application/json" \
        "https://wiki-api.yandex-team.ru/_api/frontend/kotiki/.keywords"
        %%

        Пример ответа:

        %%(js)
        {
            "data": {
                "keywords": [
                    "фото",
                    "котики"
                ]
            },
            "debug": {...},
            "user": {...}
        }
        %%
        """
        return self.build_response({'keywords': request.page.keywords_list})

    @raises(InvalidDataSentError)
    def put(self, request, *args, **kwargs):
        """
        Заменить список ключевых слов для страницы другим списком.

        Пример запроса:

        %%(sh)
        curl -H "Authorization: OAuth <token>" \
             -H "Content-Type: application/json" \
             -X PUT \
        'https://wiki-api.yandex-team.ru/_api/frontend/kotiki/.keywords' \
        -d '{"keywords": ["wow", "such page"]}'
        %%

        Пример ответа:

        %%(js)
        {
            "data": {"success": true},
            "debug": {...},
            "user": {...}
        }
        %%
        """
        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)

        keywords_logic.update_keywords(
            page=request.page,
            user=request.user,
            keywords=serializer.data['keywords'],
        )
        return self.get(request, *args, **kwargs)
